/********************************************************************************************************
 * QRNA - Comparative analysis of biological sequences 
 *         with pair hidden Markov models, pair stochastic context-free
 *        grammars, and probabilistic evolutionary  models.
 *       
 * Version 2.0.0 (JUN 2003)
 *
 * Copyright (C) 2000-2003 Howard Hughes Medical Institute/Washington University School of Medicine
 * All Rights Reserved
 * 
 *     This source code is distributed under the terms of the
 *     GNU General Public License. See the files COPYING and LICENSE
 *     for details.
 ***********************************************************************************************************/

/* evolve.c
 *
 * ER, Fri Nov 22 10:53:20 CST 2002 [St. Louis]
 * 
 * Purpose:  Given:
 *                       - a set of joint probabilities P(a,b| t) at a given evolutionary time t,
 *
 *                       - set of conditional probabilities at time zero Q_0
 *         
 *           Calculate:
 *                       - P(a,b| r*t) using a Markov chain model for evolution
 *  
 *
 * Method:   (1) conditionals: P(a,b | t) --> Q(a | b,t) 
 *            
 *           (2) Q(a | b, r*t) = Q_0 * exp{r * log[Q_0^{-1} * Q}
 *
 *           (3) Q(a | b, r*t) --> P(a,b | r*t)
 *
 *
 *
 * ConditionalsEvolved()  creates:    Q(a | b, r*t)   from    P(a,b | t)
 *
 * Joint2Joint()          converts:   P(a | b, t)     to      P(a,b | r*t)
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>
#include <float.h>

#include "funcs.h"
#include "globals.h"
#include "squid.h"
#include "structs.h"

#ifdef MEMDEBUG
#include "dbmalloc.h"
#endif

static void    adjust_prob(double *psingle, int size);
static double  calfunc(double x, double *S, double *pml, double *pmr, int L);
static double  caldfunc(double x, double *S, double *pml, double *pmr, int L);
static void    check_three_probs (double *T, double *T_zero, double *T_infty, int dim);
static void    check_reversibility(double *QL, double *QR, double *ml, double *mr, int L);
static void    check_Q_0_reversibility (double *QL, double *QR, double *ml, double *mr, int L, int hasindel);
static int     compare_freqs(double *pml, double *pmr, double *targetfreq, int L);
static int     isQconsistent(FILE *ofp, double *Q, int L, int pedantic, int verbose);
static void    islogQconsistent(FILE *ofp, double *Q_0, double *K, int Lx, int Ly, int Lz, int verbose);
static void    islogQconsistent_2(FILE *ofp, double *Q_0, double *A, double *K, int Lx, int Ly, int Lz, int verbose);

double
Cal_lambda(FILE *ofp, double *S, double *pml, double *pmr, int L, int verbose)
{
  double lambda;
  double func;
  double dfunc;
  double x = 0;
  double xnew = 1.0;
  
  while (fabs(xnew-x) > 1.0-accuracy1) {
    
    x = xnew;

    func  = calfunc (x, S, pml, pmr, L);
    dfunc = caldfunc(x, S, pml, pmr, L);
    
    if (dfunc != 0.0) xnew = x - func/dfunc;
    
  }

  lambda = xnew;
  
  return lambda;
}

double
calfunc(double x, double *S, double *pml, double *pmr, int L)
{
  double y = 0.0;
  int    i,j;

  for (i = 0; i < L; i++)
    for (j = 0; j < L; j++) 
      y += exp(x*S[i*L+j]) * pml[i] * pmr[j];

  y -= 1.0;

  return y;
}
double
caldfunc(double x, double *S, double *pml, double *pmr, int L)
{
  double y = 0.0;
  int    i,j;

  for (i = 0; i < L; i++)
    for (j = 0; j < L; j++) 
      y += S[i*L+j] * exp(x*S[i*L+j]) * pml[i] * pmr[j];

  return y;
}

/* Function: CalculateConditionalsAndMarginals()
 *
 *           ER, Wed Sep 17 10:36:04 CDT 2003 [St. Louis]
 *
 * Purpose:  Given       a LxL matrix P(i,j)  of joint probabilities, 
 *           calculate the LxL matrix Ql(i,j) of conditional probabilities.
 *           calculate the L   vecotr ml(i)   of marginal probabilities.
 *
 *           ml[i]     = \sum_k  P(i,k|t_o) 
 *
 *           Ql[i*L+j] = P(j|i, t_o) = P(i,j|t_o) / ml[i]
 *
 *           where P(i,j|t_o) are calculated according to 
 *           using a given pammodel t_o.
 *
 *           Notice that if P(i,j) are not real joint probs (ie non symmetric)
 *           this function calculates the LEFT conditionals and LEFT marginals.
 *
 * Args:     P  - LxL joint prob matrix (prealloc)
 *
 * Returns:  Ql(LxL), conditional probabilities.
 *           ml(L),   marginal probabilities.
 *           Q and ml are alocated here, freed by caller.
 */
void
CalculateConditionalsAndMarginals(FILE *ofp, double *P, double **ret_Ql, double **ret_Qr, double **ret_ml, double **ret_mr, int L, int verbose)
{
  double *ml;
  double *mr;
  double *Ql;
  double *Qr;
  double  sum;
  int     i, j;
 
  /* paranoia */
  CheckSingleProb(P, L*L);
  if (verbose) {
    fprintf(stdout, "Joint Probs\n");
    PrintProbs(stdout, P, L);
  }

  /* allocate memory
   */
  ml = (double *) MallocOrDie (sizeof(double) * L);
  mr = (double *) MallocOrDie (sizeof(double) * L);
  Ql = (double *) MallocOrDie (sizeof(double) * L * L);
  Qr = (double *) MallocOrDie (sizeof(double) * L * L);

  /* calculate LEFT/RIGHT marginals
   */
  for (i = 0; i < L; i++) {
    ml[i] = 0.0;
    mr[i] = 0.0;
  }
  
  for (i = 0; i < L; i++)
    for (j = 0; j < L; j++) {
      ml[i] += P[i*L+j];
      mr[i] += P[j*L+i];
    }
  
  CheckSingleProb(ml, L);
  CheckSingleProb(mr, L);
  
  /* calculate LEFT conditionals
   */
  for (i = 0; i < L; i++) {
    sum = 0.;
    for (j = 0; j < L; j++) {
      Ql[i*L+j] = P[i*L+j] / ml[i];
      sum += Ql[i*L+j];
    }

   /* normalize */
    for (j = 0; j < L; j++) 
      Ql[i*L+j] /= sum;
  }
   
  /* calculate RIGHT conditionals
   */
  for (i = 0; i < L; i++) {
    sum = 0.;
    for (j = 0; j < L; j++) {
      Qr[i*L+j] = P[j*L+i] / mr[i];
      sum += Qr[i*L+j];
    }
    
    /* normalize */
    for (j = 0; j < L; j++) 
      Qr[i*L+j] /= sum;
  }

  /* consistency check, rows should add up to one
   */
  for (i = 0; i < L; i++)
    CheckSingleProb(Ql+i*L, L);
  for (i = 0; i < L; i++)
    CheckSingleProb(Qr+i*L, L);
  
  if (verbose) {
    printf("RIGHT  marginalized probabilities\n");
    PrintVectorProbs(stdout, mr, L);
    printf("LEFT marginalized probabilities\n");
    PrintVectorProbs(stdout, ml, L);
    fprintf(stdout, "RIGHT Conditionals\n");
    PrintProbs(stdout, Qr, L);
    fprintf(stdout, "LEFT Conditionals\n");
    PrintProbs(stdout, Ql, L);
  }
  
  *ret_ml = ml;
  *ret_mr = mr;
  *ret_Ql = Ql;
  *ret_Qr = Qr;
}

void
ComputeConditionalsAndMarginals(FILE *ofp, double *P, double *QL, double *QR, double *pml, double *pmr, int L, int verbose)
{
  double *Ql;
  double *Qr;
  double *ml;
  double *mr;
 
  CalculateConditionalsAndMarginals(ofp, P, &Ql, &Qr, &ml, &mr, L, verbose);

   CopyMatrix (QL, Ql, L, L);
   CopyMatrix (QR, Qr, L, L);
   CopyVector (pml, ml, L);
   CopyVector (pmr, mr, L);

   free (Ql);
   free (Qr);
   free (ml);
   free (mr);
}

int
Check_Accuracy(double *vec, int L)
{
  int    i;
  int    issig;
  double m;
  double acc;

  if (L <= 5) acc = accuracy1;
  else        acc = accuracy;

  issig = FALSE;
  
  for (i = 0; i < L; i ++) 
    {
      m = (vec[i] > 0)? vec[i]:-vec[i];
      if (m > (1.-acc)/(1.*L)) {
	issig = TRUE;
	break;
      }
    }
  
  return issig;
}

/* Function: ChangeFrequencies()
 * Date:     ER, Sat Sep 27 10:05:51 CDT 2003 [St. Louis]
 *
 * Purpose:  Given a set of marginals pml, pmr change them
 *           to targetfreq
 *
 *           If this is the case of having indels as extra characters, we do that
 *           with only the subspace (L-1)x(L-1).
 *
 *  
 * Args:              
 *
 * Returns:  (void)
 *           
 */
void
ChangeFrequencies(FILE *ofp, double *Q_0, double *R, double *pm, int L, int hasindel, int verbose)
{
  double *psat;
  int     dim;
  int     i;
  
  if (hasindel) dim = L-1;
  else          dim = L;
  
  psat = SaturationProbs(R, Q_0, L, FALSE, verbose);
  
  for (i = 0; i < dim; i++) 
    if (hasindel)  pm[i] = psat[i] * (1.0 - pm[L-1]) /(1.0-psat[L-1]);
    else           pm[i] = psat[i];
  
  CheckSingleProb(pm, L);
  
  if (verbose) {
    fprintf(ofp, "New target marginals\n");
    PrintVectorProbs(ofp, pm, L);
  }
 
   free (psat);
}

/* Function: ChangeQ_0Matrix()
 * Date:     ER, Sat Sep 27 10:34:17 CDT 2003 [St. Louis]
 *
 * Purpose:  Given a matrix of conditionals at time zero Q_0,  modify the
 *           rmatrix so it stays reversible respect to the new frequecies.
 *
 *           If this is the case of having indels as extra characters, we do that
 *           with only the subspace (L-1)x(L-1).
 *  
 * Method:   
 *    
 *                    (Q_0)_ij = S_ij (p_i*p_j)^{alpja_{ij}} * p_j  
 *
 *          to  
 *                    (Q_0) _ij = S_ij (p'_i*p'_j/p_i*p_j)^{alpja_{ij}} * p'_j/p_j  
 *
 *
 * Args:              
 *
 * Returns:  (void)
 *           
 */
void
ChangeQ_0Matrix(FILE *ofp, double *Q_0, double *Q_0_inv, double *ml, double *mr, int L, double *targetfreq, 
		int hasindel, int isleft, int verbose)
{
  double           *norm;
  double           *M;
  double           *S;
  double           *Id;
  double           *pstatl;
  double           *pstatr;
  double           *alpha;
  double            comesfrom;
  double            goesto;
  double            sum;
  int               flag;
  int               dim;
  int               i,j;

  Id = Cal_Id(L);

  if (hasindel) dim = L-1;
  else          dim = L;
  
 if (verbose) {
    fprintf(ofp, "Old Q_0 Matrix\n");
    PrintProbs(ofp, Q_0, L);
 }

 flag = TRUE;
 for (i = 0; i < dim; i++)
   for (j = 0; j < dim; j++) 
     if (fabs(Q_0[i*L+j] - Id[i*L+j])  > MARGIN) flag = FALSE; 
     
 if (flag) return;

 /* paranoia */
  CheckSingleProb(ml, L);
  CheckSingleProb(mr, L);
  CheckSingleProb(targetfreq, dim);
  
  pstatl = (double *) MallocOrDie (sizeof(double) * dim);
  pstatr = (double *) MallocOrDie (sizeof(double) * dim);

  for (i = 0; i < dim; i++) {
    if (hasindel) pstatl[i] = ml[i]/(1.0-ml[L-1]);
    else          pstatl[i] = ml[i];

    if (hasindel) pstatr[i] = mr[i]/(1.0-mr[L-1]);
    else          pstatr[i] = mr[i];
  }
  
  alpha = (double *) MallocOrDie (sizeof(double) * L * L);

  for (i = 0; i < L; i++)
    for (j = 0; j < L; j++) 
      alpha[i*L+j] = -0.5;

  for (i = 0; i < dim; i++)
    for (j = i+1; j < dim; j++) {
      alpha[i*L+j] = -0.5;      
      alpha[j*L+i] = alpha[i*L+j]; 
    }

  if (hasindel) 
    for (i = 0; i < dim; i++) {
      alpha[i*L+dim] = -0.5;
      alpha[dim*L+i] = -0.5;
    }    
  
  if (verbose) {
    fprintf(ofp, "alpha Matrix\n");
    PrintProbs(ofp, alpha, L);
    fprintf(ofp, "stat L marginals\n");
    PrintVectorProbs(ofp, pstatl, L);
    fprintf(ofp, "stat R marginals\n");
    PrintVectorProbs(ofp, pstatr, L);
    fprintf(ofp, "target marginals\n");
    PrintVectorProbs(ofp, targetfreq, L);
  }
  
  S = (double *) MallocOrDie (sizeof(double) * dim * dim); 
  
  for (i = 0; i < dim; i++) 
    for (j = 0; j < dim; j++) {
      if (isleft) S[i*dim+j] = Q_0[i*L+j]/pstatr[j];
      else        S[i*dim+j] = Q_0[i*L+j]/pstatl[j];
      
      if (isleft) {
	comesfrom = -log(pstatl[i]);
	goesto    = -log(pstatr[j]);
      }
      else {
	comesfrom = -log(pstatr[i]);
	goesto    = -log(pstatl[j]);
      }
      
      S[i*dim+j] *= exp( alpha[i*L+j] * (comesfrom+goesto) );
      
    }
  
  fprintf(stdout, "S probabilities\n");
  PrintProbs(stdout, S, dim);
  
  /* New Q0  matrix */
  for (i = 0; i < dim; i++) 
    for (j = 0; j < dim; j++) {
      
      comesfrom = log(targetfreq[i]);
      goesto    = log(targetfreq[j]);
      
      Q_0[i*L+j] = S[i*dim+j] * exp(alpha[i*L+j]*comesfrom + (1.0+alpha[i*L+j])*goesto);
    }
  
  /* Normalize Q_0 */
  norm = (double *) MallocOrDie (sizeof(double) * L); 
  for (i = 0; i < L; i++) { 
    sum = 0.0;
    
    for (j = 0; j < L; j++) 
      sum += Q_0[i*L+j];

    norm[i] = 1.0/sum;
    printf("norm %d %f\n", i, sum);

    for (j = 0; j < L; j++) 
      Q_0[i*L+j] /= sum;
    
  }
  
  sum = 0.0;
  M = (double *) MallocOrDie (sizeof(double) * dim * dim); 
  for (i = 0; i < dim; i++) 
    for (j = 0; j < dim; j++) {

      comesfrom = log(targetfreq[i]);
      goesto    = log(targetfreq[j]);

      if (isleft) M[i*dim+j] = norm[i]*S[i*dim+j]*exp((1.0+alpha[i*L+j])*comesfrom + (1.0+alpha[i*L+j])*goesto); 
      else        M[i*dim+j] = norm[j]*S[j*dim+i]*exp((1.0+alpha[i*L+j])*comesfrom + (1.0+alpha[i*L+j])*goesto);

      sum += M[i*dim+j];
    }

  if (verbose) {
    fprintf(ofp, "sum %f\n", sum);
    PrintMatrix(ofp, M, dim);
  }

  if (verbose) {
    fprintf(ofp, "New Q_0 Matrix\n");
    PrintMatrix(ofp, Q_0, L);
  }

  /* consistency check, rows should add up to one
   */
  for (i = 0; i < L; i++)
    CheckSingleProb(Q_0+i*L, L);
  
  Comp_M_Inv(ofp, Q_0, Q_0_inv, L, verbose);

  free(norm);
  free(M);
  free(S);
  free(Id);
  free(pstatl);
  free(pstatr);
  free(alpha);
}

/* Function: ChangeQ_0MatrixNaive()
 * Date:    on Sep 29 18:25:19 CDT 2003  [St. Louis]
 *
 * Purpose:  Given a matrix of conditionals at time zero Q_0,  modify the
 *           rmatrix so it stays reversible respect to the new frequecies.
 *
 *           If this is the case of having indels as extra characters, we do that
 *           with only the subspace (L-1)x(L-1).
 *  
 * Method:   
 *    
 *
 *
 * Args:              
 *
 * Returns:  (void)
 *           
 */
void
ChangeQ_0MatrixNaive(FILE *ofp, double *QL_0, double *QR_0, int L, double *targetfreq, int hasindel, int isleft, int verbose)
{
  double           *P;
  double           *Id;
  double           *Qnewl;
  double           *Qnewr;
  double           *pml;
  double           *pmr;
  int               flag;
  int               dim;
  int               i,j;

  Id = (double *)Cal_Id(L);

  if (hasindel) dim = L-1;
  else          dim = L;
  
 if (verbose) {
    fprintf(ofp, "Old QL_0 Matrix\n");
    PrintProbs(ofp, QL_0, L);
    fprintf(ofp, "Old QR_0 Matrix\n");
    PrintProbs(ofp, QR_0, L);
 }

 flag = TRUE;
 for (i = 0; i < dim; i++)
   for (j = 0; j < dim; j++) 
     if (fabs(QL_0[i*L+j] - Id[i*L+j])  > MARGIN) flag = FALSE; 
     
 if (flag) return;

 /* paranoia */
  CheckSingleProb(targetfreq, dim);
  
  P = (double *) MallocOrDie (sizeof(double) * dim * dim);

  for (i = 0; i < dim; i++)
    for (j = 0; j < dim; j++) 
      P[i*dim+j] = targetfreq[i] * 0.5 * (QL_0[i*L+j] + QR_0[j*L+i]);

 if (verbose) {
    fprintf(ofp, "new P Matrix\n");
    PrintProbs(ofp, P, dim);
 }
  
  CalculateConditionalsAndMarginals(ofp, P, &Qnewl, &Qnewr, &pml, &pmr, dim, verbose);
  fprintf(ofp, "Naive newQl(i|j,t) probabilities\n");
  PrintProbs(ofp, Qnewl, dim);
  fprintf(ofp, "Naive newQr(i|j,t) probabilities\n");
  PrintProbs(ofp, Qnewr, dim);
  fprintf(ofp, "Naive marginals probabilities\n");
  PrintVectorProbs(ofp, pml, dim);
  PrintVectorProbs(ofp, pmr, dim);
  
  free(Qnewl);
  free(Qnewr);
  free(pml);
  free(pmr);
  
  free(P);
  free(Id);
}

void
ChangeQ_0MatrixIterate(FILE *ofp, double *Q_0, double *Q_0_inv, double *pml, double *pmr, int L, double *targetfreq, 
		       int hasindel, int isleft, int verbose)
{
  double           *P;
  double           *S;
  double           *Ql;
  double           *Qr;
  double           *Id;
  double           *pstatl;
  double           *pstatr;
  double            lambda;
  int               flag;
  int               dim;
  int               iterations = 0;
  int               i,j;

  if (hasindel) dim = L-1;
  else          dim = L;
  
 Id = Cal_Id(L);

  flag = TRUE;
  for (i = 0; i < dim; i++)
    for (j = 0; j < dim; j++) 
      if (fabs(Q_0[i*L+j] - Id[i*L+j])  > MARGIN) flag = FALSE; 
  if (flag) { free(Id); return;}
  
  /* paranoia */
  CheckSingleProb(pml, L);
  CheckSingleProb(pmr, L);
  CheckSingleProb(targetfreq, dim);
  
  P      = (double *) MallocOrDie (sizeof(double) * dim * dim); 
  Ql     = (double *) MallocOrDie (sizeof(double) * dim * dim); 
  Qr     = (double *) MallocOrDie (sizeof(double) * dim * dim); 
  S      = (double *) MallocOrDie (sizeof(double) * dim * dim); 
  pstatl = (double *) MallocOrDie (sizeof(double) * dim);
  pstatr = (double *) MallocOrDie (sizeof(double) * dim);


  for (i = 0; i < dim; i++) {
    if (hasindel) pstatl[i] = pml[i]/(1.0-pml[L-1]);
    else          pstatl[i] = pml[i];

    if (hasindel) pstatr[i] = pmr[i]/(1.0-pmr[L-1]);
    else          pstatr[i] = pmr[i];
  }

  for (i = 0; i < dim; i++)
    for (j = 0; j < dim; j++) 
      if (isleft) { Ql[i*dim+j] = Q_0[i*L+j]; Qr[i*dim+j] = 0.0; }
      else        { Qr[i*dim+j] = Q_0[i*L+j]; Ql[i*dim+j] = 0.0; }

  if (isleft)
    Joint_From_Condi(ofp, P, Ql, pstatl, dim, verbose);
  else
    Joint_From_Condi(ofp, P, Qr, pstatr, dim, verbose);

  if (verbose) {
    fprintf(ofp, "Old Ql_0(i|j,t) probabilities\n");
    PrintProbs(ofp, Ql, dim);
    fprintf(ofp, "Old Qr_0(i|j,t) probabilities\n");
    PrintProbs(ofp, Qr, dim);
    fprintf(ofp, "Old marginals probabilities\n");
    PrintVectorProbs(ofp, pml, L);
    PrintVectorProbs(ofp, pmr, L);
    fprintf(ofp, "Old P_0 Matrix\n");
    PrintProbs(ofp, P, dim);
  }

  while (compare_freqs(pstatl, pstatr, targetfreq, dim)) {
    iterations ++;

    for (i = 0; i < dim; i++) 
      for (j = 0; j < dim; j++) 
	S[i*dim+j] = log(P[i*dim+j]) - log(pstatl[i]) - log(pstatr[j]);
    
    lambda = Cal_lambda(ofp, S, targetfreq, targetfreq, dim, verbose);
    
    for (i = 0; i < dim; i++) 
      for (j = 0; j < dim; j++) 
	P[i*dim+j] = exp(lambda*S[i*dim+j])*targetfreq[i]*targetfreq[j];
    
    /* check they are probabilities */  
    CheckSingleProb(P, dim*dim);
    
    ComputeConditionalsAndMarginals(ofp, P, Ql, Qr, pstatl, pstatr, dim, verbose);
  }
  
  if (verbose) {
    fprintf(ofp, "new P_0 (%d it) Matrix\n", iterations);
    PrintProbs(ofp, P, dim);
    fprintf(ofp, "new Ql_0(i|j,t) probabilities\n");
    PrintProbs(ofp, Ql, dim);
    fprintf(ofp, "new Qr_0(i|j,t) probabilities\n");
    PrintProbs(ofp, Qr, dim);
    fprintf(ofp, "new marginals probabilities\n");
    PrintVectorProbs(ofp, pstatl, dim);
    PrintVectorProbs(ofp, pstatr, dim);  
  }
  
    for (i = 0; i < dim; i++) 
      for (j = 0; j < dim; j++) 
	if (isleft) Q_0[i*L+j] = Ql[i*dim+j];
	else        Q_0[i*L+j] = Qr[i*dim+j];

  /* consistency check, rows should add up to one
   */
  for (i = 0; i < L; i++)
    CheckSingleProb(Q_0+i*L, L);
  
  Comp_M_Inv(ofp, Q_0, Q_0_inv, L, verbose);

  free(Ql);
  free(Qr);
  free(pstatl);
  free(pstatr);
  
  free(Id);
  free(S);
  free(P);
}

/* Function: ChangePairProbs()
 * Date:     ER, Wed Mar  5 10:04:54 CST 2003 [St. Louis]
 *
 * Purpose: To change the overall base composition, [we do this before dealing with gaps]
 *
 * calculate:         P(i|j) = P(i,j) / P(j)
 *
 * then use:         ^P(j) = f(j)
 *
 * to recalculate:   ^P(i,j) = P(i|j) * ^P(j)
 *
 *
 *
 */
void
ChangePairProbs (int L5, double *pair5prob, double *psingle_target, int verbose)
{
  double *pair5cond;              /* [5x5] base-pair conditional probabilities +*/
  double *pmar;                    /* [5] marginalization of the pair5probs +*/
  int     i,j;                      /* Symbols for emission prob's              +*/
  int     L5SQ = L5*L5;

  if (verbose) {
    fprintf(stdout, "Target Marginal probabilities \n");
    PrintVectorProbs(stdout, psingle_target, 5);
  }

  if (verbose) {
    fprintf(stdout, "Ppair5(i,j) Unsym--Joint probabilities \n");
    PrintProbs(stdout, pair5prob, L5);
  }

  /* First, symmetrise the matrix, otherwise we cannot play
   * this trick
   */
  for (i = 0; i < L5; i++)
    for (j = i+1; j < L5; j++)
      if (pair5prob[idx5(i,j)] != pair5prob[idx5(j,i)]) {

        pair5prob[idx5(i,j)] = 0.5 * ( pair5prob[idx5(i,j)] + pair5prob[idx5(j,i)] );
        pair5prob[idx5(j,i)] = pair5prob[idx5(i,j)];

      }
   CheckSingleProb(pair5prob, L5SQ);

  pair5cond = (double *) MallocOrDie (sizeof(double) * L5SQ);
  pmar      = (double *) MallocOrDie (sizeof(double) * L5);

  MarginalizeJointProbs(pair5prob, pmar, L5, 2);

  /* these loops go just to 3, because we are not dealing with gaps just yet */
  for (i = 0; i < L5SQ; i++)
    pair5cond [i] = 0.0;

  for (i = 0; i < L5; i++)
    for (j = 0; j < L5; j++)
      if (pmar[j] > 0.0) pair5cond[idx5(i,j)] = pair5prob[idx5(i,j)] / pmar[j];
      else               pair5cond[idx5(i,j)] = 0.0;

  if (verbose) {
    fprintf(stdout, "Ppair5(i,j) Before--Joint probabilities \n");
    PrintProbs(stdout, pair5prob, L5);
    fprintf(stdout, "Ppair5(i,j) Conditionals probabilities \n");
    PrintProbs(stdout, pair5cond, L5);
    fprintf(stdout, "Marginal probabilities \n");
    PrintVectorProbs(stdout, pmar, L5);
  }

  for (i = 0; i < L5; i++)
    for (j = 0; j < L5; j++)
      pair5prob[idx5(i,j)] = pair5cond[idx5(i,j)] * psingle_target[j];

  DNorm(pair5prob, L5SQ);

  if (verbose) {
    fprintf(stdout, "Ppair5(i,j) After--Joint probabilities \n");
    PrintProbs(stdout, pair5prob, L5);

    fprintf(stdout, "Ppair5 Marginal probabilities \n");
    MarginalizeJointProbs(pair5prob, pmar, L5, 2);
    PrintVectorProbs(stdout,  pmar, L5);
  }

  free(pmar);
  free(pair5cond);
}

void
ChangePairProbsIterate(FILE *ofp, double *pairprobs, int L, double *targetfreq, int verbose)
{
  double           *S;
  double           *Ql;
  double           *Qr;
  double           *pml;
  double           *pmr;
  double            lambda;
  int               iterations = 0;
  int               i,j;

  CalculateConditionalsAndMarginals(ofp, pairprobs, &Ql, &Qr, &pml, &pmr, L, verbose);

  if (verbose) {
    fprintf(ofp, "Old PAIR Matrix\n");
    PrintProbs(ofp, pairprobs, L);
    fprintf(ofp, "Old Ql(i|j,t) probabilities\n");
    PrintProbs(ofp, Ql, L);
    fprintf(ofp, "Old Qr(i|j,t) probabilities\n");
    PrintProbs(ofp, Qr, L);
    fprintf(ofp, "Old marginals probabilities\n");
    PrintVectorProbs(ofp, pml, L);
    PrintVectorProbs(ofp, pmr, L);
  }
  
  S = (double *) MallocOrDie (sizeof(double) * L * L); 
  
  while (compare_freqs(pml, pmr, targetfreq, L)) {
    iterations ++;

    for (i = 0; i < L; i++) 
      for (j = 0; j < L; j++) 
	S[i*L+j] = log(pairprobs[i*L+j]) - log(pml[i]) - log(pmr[j]);
    
    lambda = Cal_lambda(ofp, S, targetfreq, targetfreq, L, verbose);
    
    for (i = 0; i < L; i++) 
      for (j = 0; j < L; j++) 
	pairprobs[i*L+j] = exp(lambda*S[i*L+j])*targetfreq[i]*targetfreq[j];
    
    /* check they are probabilities */  
    CheckSingleProb(pairprobs, L*L);
    
    ComputeConditionalsAndMarginals(ofp, pairprobs, Ql, Qr, pml, pmr, L, verbose);
  }
  
  if (fabs(lambda) < MARGIN) Die ("ChangePairProbsIterate(): lambda = 0. trivial");

  if (verbose) {
    fprintf(ofp, "new PAIR (%d it) Matrix\n", iterations);
    PrintProbs(ofp, pairprobs, L);
    fprintf(ofp, "new Ql(i|j,t) probabilities\n");
    PrintProbs(ofp, Ql, L);
    fprintf(ofp, "new Qr(i|j,t) probabilities\n");
    PrintProbs(ofp, Qr, L);
    fprintf(ofp, "new marginals probabilities\n");
    PrintVectorProbs(ofp, pml, L);
    PrintVectorProbs(ofp, pmr, L);  
  }
  
  free(Ql);
  free(Qr);
  free(pml);
  free(pmr);
  
   free(S);
}

/* Function: ChangeRateMatrix()
 * Date:     ER, Wed Sep 17 15:38:25 CDT 2003 [St. Louis]
 *
 * Purpose:  Given a rate matrix R,  modify the
 *           rate matrix so it stays reversible respect to the new frequecies.
 *
 *           If this is the case of having indels as extra characters, we do that
 *           with only the subspace (L-1)x(L-1).
 *  
 * Method:   
 *    
 *                    (Q_0 K)_ij = S_ij (p_j/p_i)^{1/2}  i \neq j 
 *
 *          to  
 *                    (Q_0 K') _ij = S_ij (p'_j/p'_i)^{1/2}
 * *          so         K' = Q_0_inv * (Q_0 K')
 *
 * Args:              
 *
 * Returns:  (void)
 *           
 */
void
ChangeRateMatrix(FILE *ofp, double *Q_0R, double *Q_0_inv, double *R, double *ml, double *mr, int L, double *targetfreq, 
		 int hasindel, int isleft, int verbose)
{
  double *pstatl;
  double *pstatr;
  double *alpha;
  double *Rev;
  double  comesfrom;
  double  goesto;
  double  sum;
  int     dim;
  int     i,j;

  if (hasindel) dim = L-1;
  else          dim = L;
  
 if (verbose) {
    fprintf(ofp, "Old Rate Matrix\n");
    PrintProbs(ofp, R, L);
 }

  /* paranoia */
  CheckSingleProb(ml, L);
  CheckSingleProb(mr, L);
  CheckSingleProb(targetfreq, dim);
  
  pstatl = (double *) MallocOrDie (sizeof(double) * dim);
  pstatr = (double *) MallocOrDie (sizeof(double) * dim);

  for (i = 0; i < dim; i++) {
    if (hasindel) pstatl[i] = ml[i]/(1.0-ml[L-1]);
    else          pstatl[i] = ml[i];

    if (hasindel) pstatr[i] = mr[i]/(1.0-mr[L-1]);
    else          pstatr[i] = mr[i];
  }
  
  alpha = (double *) MallocOrDie (sizeof(double) * L * L);

  for (i = 0; i < L; i++)
    for (j = 0; j < L; j++) 
      alpha[i*L+j] = 0.0;

  for (i = 0; i < dim; i++)
    for (j = i+1; j < dim; j++) {
      alpha[i*L+j] = -0.5;      
      alpha[j*L+i] = alpha[i*L+j]; 
    }

  if (hasindel) 
    for (i = 0; i < dim; i++) {
      alpha[i*L+dim] = -0.5;
      alpha[dim*L+i] = -0.5;
    }    

 if (verbose) {
    fprintf(ofp, "alpha Matrix\n");
    PrintProbs(ofp, alpha, L);
    fprintf(ofp, "stat L marginals\n");
    PrintVectorProbs(ofp, pstatl, L);
    fprintf(ofp, "stat R marginals\n");
    PrintVectorProbs(ofp, pstatr, L);
    fprintf(ofp, "target marginals\n");
    PrintVectorProbs(ofp, targetfreq, L);
  }
 
 /* Non diagonal "reduced" components of (Q_0*R) */
  for (i = 0; i < dim; i++) 
    for (j = 0; j < dim; j++) {

      if (isleft) {
	comesfrom = targetfreq[i]/pstatl[i];
	goesto    = targetfreq[j]/pstatr[j];
      }
      else {
	comesfrom = targetfreq[i]/pstatr[i];
	goesto    = targetfreq[j]/pstatl[j];
      }
      
     if (i != j ) Q_0R[i*L+j] *= EXP2(alpha[i*L+j]*LOG2(comesfrom)) * EXP2((1.0+alpha[i*L+j])*LOG2(goesto));
    }

  if (hasindel) {
    for (i = 0; i < dim; i++) {

      if (isleft) {
	comesfrom = targetfreq[i]/pstatl[i];
	goesto    = targetfreq[i]/pstatr[i];
      }
      else {
	comesfrom = targetfreq[i]/pstatr[i];
	goesto    = targetfreq[i]/pstatl[i];
      }

      Q_0R[i*L+dim] *= EXP2(alpha[i*L+dim]      *LOG2(comesfrom));
      Q_0R[dim*L+i] *= EXP2((1.0+alpha[dim*L+i])*LOG2(goesto));
    }

  }

  /* Diagonal components of (Q_0*R) */
  for (i = 0; i < L; i++) { 
    Q_0R[i*L+i] = 0.0;
    
    for (j = 0; j < L; j++) 
      if (j != i) Q_0R[i*L+i] -= Q_0R[i*L+j];
  }
 
  if (verbose) {
    Rev = (double *) MallocOrDie (sizeof(double) * dim * dim);
    fprintf(ofp, "(Q_0*R * pi) reversibility\n");
    for (i = 0; i < dim; i++) 
      for (j = 0; j < dim; j++)
	Rev[i*dim+j] = Q_0R[i*L+j] * targetfreq[i];
	
    PrintProbs(ofp, Rev, dim);
    free(Rev);
  }

  Comp_M_N_Prod(ofp, Q_0_inv, Q_0R, L, verbose); /*  multiply Q_0_inv*M it in M */
  
  CopyMatrix(R, Q_0R, L, L);

  /* consistency check, rows should add up to zero
   */	
  for (i = 0; i < L; i++) {
    sum = 0.0;
    for (j = 0; j < L; j++)
      sum += R[i*L+j];
    if (sum > MARGIN || sum < -MARGIN) Warn("ChangeRateMatrix(): column %d bad rate matrix (sum = %f)\n", i, sum);
  }
  
  if (verbose) {
    fprintf(ofp, "New Rate Matrix\n");
    PrintProbs(ofp, R, L);
  }

  free(pstatl);
  free(pstatr);
  free(alpha);
}

/* Function: ChangeRIBORateMatrix()
 * Date:     ER, Wed Sep 17 15:38:25 CDT 2003 [St. Louis]
 *
 * Purpose:  Given a rate matrix R,  modify the
 *           rate matrix so it stays reversible respect to the new frequecies.
 *
 *           If this is the case of having indels as extra characters, we do that
 *           with only the subspace (L-1)x(L-1).
 *  
 * Method:   
 *    
 *                    (Q_0 K)_ij = S_ij (p_j/p_i)^{1/2}  i \neq j 
 *
 *          to  
 *                    (Q_0 K') _ij = S_ij (p'_j/p'_i)^{1/2}
 * *          so         K' = Q_0_inv * (Q_0 K')
 *
 * Args:              
 *
 * Returns:  (void)
 *           
 */
void
ChangeRIBORateMatrix(FILE *ofp, double *R, double *Q_0_inv, double *pm, int L, double *targetpair, int verbose)
{
  double *alpha;
  double *Rev1, *Rev2;
  double *epm;
  double  comesfrom;
  double  goesto;
  double  sum;
  int     L2;
  int     i,j;

  L2 = L * L;
 
  if (verbose) {
    fprintf(ofp, "Old Rate Matrix\n");
    PrintProbs(ofp, R, L2);
    fprintf(ofp, "old stat probabilities\n");
    PrintProbs(ofp, pm, L);
    fprintf(ofp, "target stat probabilities\n");
    PrintProbs(ofp, targetpair, L);
 }

  /* paranoia */
  CheckSingleLog2Prob(pm,         L2);
  CheckSingleLog2Prob(targetpair, L2);
  
  alpha = (double *) MallocOrDie (sizeof(double) * L2 * L2);

  for (i = 0; i < L2; i++)
    for (j = 0; j < L2; j++) 
      alpha[i*L2+j] = 0.0;

  for (i = 0; i < L2; i++)
    for (j = i+1; j < L2; j++) {
      alpha[i*L2+j] = -0.5;      
      alpha[j*L2+i] = alpha[i*L2+j]; 
    }

  if (verbose) {
    fprintf(ofp, "alpha Matrix\n");
    PrintProbs(ofp, alpha, L2);
    fprintf(ofp, "stat marginals\n");
    PrintVectorProbs(ofp, pm, L2);
    fprintf(ofp, "target marginals\n");
    PrintVectorProbs(ofp, targetpair, L2);
  }
 
 /* Non diagonal "reduced" components of (R) */
  for (i = 0; i < L2; i++) 
    for (j = 0; j < L2; j++) {

   	comesfrom = targetpair[i] - pm[i];
	goesto    = targetpair[j] - pm[j];
      
     if (i != j ) R[i*L2+j] *= EXP2(alpha[i*L2+j]*comesfrom) * EXP2((1.0+alpha[i*L2+j])*goesto);
    }

  /* Diagonal components of (Q_0*R) */
  for (i = 0; i < L2; i++) { 
    R[i*L2+i] = 0.0;
    
    for (j = 0; j < L2; j++) 
      if (j != i) R[i*L2+i] -= R[i*L2+j];
  }
 
  if (verbose) {
    Rev1 = (double *) MallocOrDie (sizeof(double) * L2 * L2);
    Rev2 = (double *) MallocOrDie (sizeof(double) * L2 * L2);
    for (i = 0; i < L2; i++) 
      for (j = 0; j < L2; j++) {
	Rev1[i*L2+j] = R[i*L2+j] * EXP2(targetpair[i]);
	Rev2[i*L2+j] = R[j*L2+i] * EXP2(targetpair[j]);

	if (fabs(Rev1[i*L2+j]-Rev2[i*L2+j]) > MARGIN) 
	  Die ("ChangeRIBORateMatrix(): New rate matrix is not reversible respect to the new probabilities");
      }
	
    fprintf(ofp, "(R * pi) reversibility\n");
    PrintProbs(ofp, Rev1, L2);
    fprintf(ofp, "(R * pi) reversibility\n");
    PrintProbs(ofp, Rev2, L2);
    free(Rev1);
    free(Rev2);
  }

  Comp_M_N_Prod(ofp, Q_0_inv, R, L2, verbose); /*  multiply Q_0_inv*M it in M */
  
  /* consistency check, rows should add up to zero
   */	
  for (i = 0; i < L2; i++) {
    sum = 0.0;
    for (j = 0; j < L2; j++)
      sum += R[i*L2+j];
    if (sum > MARGIN || sum < -MARGIN) Warn("ChangeRIBORateMatrix(): column %d bad rate matrix (sum = %f)\n", i, sum);
  }

  for (i = 0; i < L2; i++) 
    pm[i] = targetpair[i];
  
  if (verbose) {
    epm = (double *) MallocOrDie (sizeof(double) * L2);

    for (i = 0; i < L2; i++) 
      epm[i] = EXP2(targetpair[i]);
    
    fprintf(ofp, "New RIBO Rate Matrix\n");
    PrintProbs(ofp, R, L2);
    fprintf(ofp, "New RIBO MARGINALS Matrix\n");
    PrintProbs(ofp, epm, L);
    
    free(epm);  
  }
  
}

/* Function: Condi_From_Joint()
 * Date:     ER, Wed Mar  1 11:04:24 CST 2000 [St. Louis]
 *
 * Purpose:  Given       a LL matrix M(i,j) of joint probabilities, 
 *           calculate the LL matrix Q(i,j) of conditional probabilities.
 *
 *           Q[i*L+j] = Q(j|i) = P(i,j|t_o) / \sum_k  P(i,k|t_o) 
 *
 *           where P(i,j|t_o) are calculated according to 
 *           using a given pammodel t_o.
 *
 *           Notice that if P(i,j) are not real joint probs (ie non symmetric)
 *           this function calculates the RIGHT conditionals and LEFT marginals.
 *
 * Args:     P  - LxL joint prob matrix (prealloc)
 *
 * Returns:  Q(LxL), conditional probabilities.
 *           Q is alocated here, freed by caller.
 */
double *
Condi_From_Joint(FILE *ofp, double *P, int L, int verbose)
{
  double *Q;
  double *pm;
  double  sum;
  int     i, j;
  
  if (verbose) {
    fprintf(ofp, "P(i,j, t_o) probabilities\n");
    PrintProbs(ofp, P, L);
  }
  
  /* allocate Q[L*L]
   */
  Q  = (double *) MallocOrDie (sizeof(double) * L * L);
  pm = (double *) MallocOrDie (sizeof(double) * L);

  for (i = 0; i < L; i++)
    pm[i] = 0.0;
  
  for (i = 0; i < L; i++)
    for (j = 0; j < L; j++) 
      pm[i] += P[i*L+j];
  
  for (i = 0; i < L; i++) {
    sum = 0.;
    for (j = 0; j < L; j++) {
      Q[i*L+j] = P[i*L+j] / pm[i];
      sum += Q[i*L+j];
    }
    /* normalize */
    for (j = 0; j < L; j++) 
      Q[i*L+j] /= sum;
  }
 
  if (verbose) { 
    fprintf(ofp, "Q(j|i, t_o) probabilities\n");
    PrintProbs(ofp, Q, L);
  }
  
  /* consistency check, rows should add up to one
   */
  for (i = 0; i < L; i++)
    CheckSingleProb(Q+i*L, L);
  
  free (pm);

  return Q;
}

double *
CondiR_From_Joint(FILE *ofp, double *P, int L, int verbose)
{
  double *QR;
  double *pmr;
  double  sum;
  int     i, j;
  
  if (verbose) {
    fprintf(ofp, "P(i,j, t_o) probabilities\n");
    PrintProbs(ofp, P, L);
  }
  
  /* allocate Q[L*L]
   */
  QR  = (double *) MallocOrDie (sizeof(double) * L * L);
  pmr = (double *) MallocOrDie (sizeof(double) * L);

  for (i = 0; i < L; i++)
    pmr[i] = 0.0;
  
  for (i = 0; i < L; i++)
    for (j = 0; j < L; j++) 
      pmr[i] += P[j*L+i];
  
  for (i = 0; i < L; i++) {
    sum = 0.;
    for (j = 0; j < L; j++) {
      QR[i*L+j] = P[j*L+i] / pmr[i];
      sum += QR[i*L+j];
    }
    /* normalize */
    for (j = 0; j < L; j++) 
      QR[i*L+j] /= sum;
  }
 
  if (verbose) { 
    fprintf(ofp, "QR(j|i, t_o) probabilities\n");
    PrintProbs(ofp, QR, L);
  }
  
  /* consistency check, rows should add up to one
   */
  for (i = 0; i < L; i++)
    CheckSingleProb(QR+i*L, L);
  
  free (pmr);

  return QR;
}

/* Function: Cond_From_Rate()
 * Date:     ER, Thu Sep 25 09:38:36 CDT 2003 [St. Louis]
 *
 * Purpose:  Given a Rate R 
 *           and set of conditional probabilities at time zero Q_0
 *         
 *           calculate coditionals; Q(t) = Q_0 * exp(tR)
 *  
 * Args:              
 *
 * Returns:  void
 *           
 */
void
Condi_From_Rate(double **ret_Q, double *R, double *Q_0, double tfactor, int L, int pedantic, int verbose)
{
  double *Qr;
  double *Q;
  double  tmax;
  double  xtime;
  int     n_times;
  int     i, j;
  
  tmax = TMAX;

  n_times = (int)tfactor / (int)tmax;
  xtime   = tfactor - tmax*(double)n_times;

  /* calculate Q(t) = Q_0 * exp{tfactor*K}
   */
  Q = (double *)Cal_Id(L);
  while (n_times > 0) {
    Qr = Cal_M_Exp(stdout, R, L, tmax, verbose);
    Comp_M_N_Prod(stdout, Qr, Q, L, FALSE); /*  multiply Qr^(n_times*tmax), dump it in Qp */
    n_times --;
    free (Qr);
  }
  Qr = Cal_M_Exp(stdout, R, L, xtime, verbose);
  Comp_M_N_Prod(stdout, Qr, Q, L, verbose); /*  multiply Qr^xtime, dump it in Qp */
 
  Comp_M_N_Prod(stdout, Q_0, Q, L, verbose); /* multiply Q_0*Qp, dump it in Qp */
  
  for (i = 0; i < L; i++)
    adjust_prob(Q+i*L, L);
  
  /* consistency check, rows should add up to one
   */	
  for (i = 0; i < L; i++)
    CheckSingleProb(Q+i*L, L);
  
  *ret_Q = Q;

  free (Qr);
}

void
ZCondi_From_Rate(double **ret_Q, struct zmatrix_s *R, struct zmatrix_s *Q_0, double tfactor, int L, int pedantic, int verbose)
{
  struct zmatrix_s *Qr;
  struct zmatrix_s *Q;
  double           *Qreal;
  double            tmax;
  double            xtime;
  int               n_times;
  int               i,j;
  
  tmax = TMAX;

  n_times = (int)tfactor / (int)tmax;
  xtime   = tfactor - tmax*(double)n_times;

  /* calculate Q(t) = Q_0 * exp{tfactor*K}
   */
  Q = (struct zmatrix_s *)ZCal_Id(L);
  while (n_times > 0) {
    Qr = ZCal_M_Exp(stdout, R, L, tmax, verbose);
    ZComp_M_N_Prod(stdout, Qr, Q, L, FALSE); /*  multiply Qr^(n_times*tmax), dump it in Qp */
    n_times --;
    FreeZmatrix (Qr);
  }
  Qr = ZCal_M_Exp(stdout, R, L, xtime, verbose);
  ZComp_M_N_Prod(stdout, Qr, Q, L, verbose); /*  multiply Qr^xtime, dump it in Qp */
 
  ZComp_M_N_Prod(stdout, Q_0, Q, L, verbose); /* multiply Q_0*Qp, dump it in Qp */
  
  if (verbose) {
    fprintf(stdout, "Conditional Zmatrix\n");
    PrintZmatrix(stdout, Q, L);
  }

  /* Is the conditional matrix real ?
   */
  for (i = 0; i < L; i++)
    for (j = 0; j < L; j++)
      if (fabs(Q->imag[i*L+j]) > MARGIN2) 
	Die ("ZCondi_From_Rate(): your conditional matrix is not real!");
 
  Qreal = (double *) MallocOrDie (sizeof(double) * L * L); 
  CopyMatrix (Qreal, Q->real, L, L);

  for (i = 0; i < L; i++)
    adjust_prob(Qreal+i*L, L);

   /* consistency check, rows should add up to one
   */	
  for (i = 0; i < L; i++)
    CheckSingleProb(Qreal+i*L, L);
  
 
  *ret_Q = Qreal;

  FreeZmatrix(Q);
  FreeZmatrix(Qr);
}


/* Function: ConditionalsEvolved()
 * Date:     ER, Wed Apr 12 18:11:23 CDT 2000 [St. Louis]
 *
 * Purpose:  Given a set of conditionals probabilities Q(a|b, t) 
 *           at a given evolutionary time t, 
 *           and set of conditional probabilities at time zero Q_0
 *         
 *           calculate Q(a,b| r*t) using a Markov chain model for evolution
 *  
 * Method:   Q(a | b, r*t) = Q_0 * exp{r * log[Q_0^{-1} * Q}
 *
 *
 *
 *
 * Args:              
 *
 * Returns: Qr(LxL). 
 *          Qr is allocated here, freed by caller.
 *           
 */
void
ConditionalsEvolved(FILE *ofp, double *Q, double *QR, double *Q_0, double *QR_0, double *ml, double *mr, 
		    int L, double tfactor, double *targetfreq, int changefreq, int hasindel, int pedantic, int verbose)
{
  double  *Q_0_inv;
  double  *QR_0_inv;
  double  *Q_0K;
  double  *QR_0KR;
  double  *Qp;
  double  *QRp;
  double  *K;
  double  *KR;
  double  *S;
  double   tmax;
  double   xtime;
  int      n_times;
  int      i;

  tmax = TMAX;

  n_times = (int)tfactor / (int)tmax;
  xtime   = tfactor - tmax*(double)n_times;

  /* consistency check, rows should add up to one
   */	
  for (i = 0; i < L; i++)  CheckSingleProb(Q+i*L, L);
  
 if (verbose) {
    fprintf(ofp, "QL(j|i, t_*)  probabilities\n");
    PrintProbs(ofp, Q, L);
    fprintf(ofp, "QR(j|i, t_*)  probabilities\n");
    PrintProbs(ofp, QR, L);
    fprintf(ofp, "QL_0(j|i)  probabilities\n");
    PrintProbs(ofp, Q_0, L);
    fprintf(ofp, "QR_0(j|i)  probabilities\n");
    PrintProbs(ofp, QR_0, L);
    fprintf(ofp, "LEFT Marginal  probabilities\n");
    PrintVectorProbs(ofp, ml, L);
    fprintf(ofp, "RIGHT  Marginal  probabilities\n");
    PrintVectorProbs(ofp, mr, L);  
  }
 
 if (verbose) {
   check_Q_0_reversibility(Q_0, QR_0, ml, mr, L, hasindel);
   check_reversibility    (Q,   QR,   ml, mr, L);
 }
 
 /* calculate Q_0_inv
  */
  Q_0_inv  = Cal_M_Inv(ofp, Q_0,  L, verbose);
  QR_0_inv = Cal_M_Inv(ofp, QR_0, L, verbose);

  /* construct K = log [Q_0_inv * Q]
   */
  Comp_M_N_Prod(ofp, Q_0_inv,  Q,  L, verbose); /*  multiply Q_0_inv*Q, dump it in Q */
  Comp_M_N_Prod(ofp, QR_0_inv, QR, L, verbose); /*  multiply Q_0_inv*Q, dump it in Q */

  S = Cal_Id(L);
  Comp_M_N_Sum(ofp, Q, S, L, L, FALSE, verbose);   /* S = Q - Id  */
  
  /* check that Q is consistent with a markovian stationary 
   * model of evolution. 
   * In plain words it checks whether logQ is going to exist.
   *
   * Basically, it checks whether  all eigenvalues of Q - I
   * are between -1 and 1. 
   *
   * remember log (1+x) = x - x^2/2 + ...   converges for -1 < x <= 1
   *
   */
  
  if (verbose) isQconsistent(ofp, S, L, pedantic, verbose);
  K  = Cal_M_Taylor_Log(ofp, Q,  L, verbose);
  KR = Cal_M_Taylor_Log(ofp, QR, L, verbose);
  
  /* if (isQconsistent(ofp, S, L, pedantic, verbose)) { 
     K  = Cal_M_Taylor_Log(ofp, Q,  L, verbose);
     KR = Cal_M_Taylor_Log(ofp, QR, L, verbose);
     }
     else {
     K  = Cal_M_Vandermonde_Log(ofp, Q,  L, verbose);
     KR = Cal_M_Vandermonde_Log(ofp, QR, L, verbose);
     }*/
  
  if (verbose) {
    fprintf(ofp, "R  rate matrix\n");
    PrintProbs(ofp, K, L);
  }
 
  if (pedantic)
    islogQconsistent(ofp, Q_0, K, L, L, L, verbose); /*  check that K is consistent with a 
						      *  markovian stationary model of evolution. 
						      *
						      *  Basically, it checks whether all entries of  
						      *  Q_0 + epsilon*Q_0*K are none negatives.
						      */

  /* If we are told to change the stationary frequencies we do it here.
   * This will also modify the rate matrix K, while keeping the exchangeability parameters S
   * defined as
   *            (Q_0 K)_ij  = S_ij (p_j/p_i)^{1/2}  i \neq j 
   *
   *   to  
   *            (Q_0 K')_ij = S_ij (p'_j/p'_i)^{1/2}
   *
   *    so       K' = Q_0_inv * (Q_0 K')
   *
   */
  if (changefreq) {
    
    Q_0K   = Cal_M_N_Prod(ofp, Q_0,  K,  L, L, L, verbose);
    QR_0KR = Cal_M_N_Prod(ofp, QR_0, KR, L, L, L, verbose);
    
    ChangeQ_0MatrixIterate(ofp, Q_0,  Q_0_inv,  ml, mr, L, targetfreq, hasindel, TRUE,  verbose);
    ChangeQ_0MatrixIterate(ofp, QR_0, QR_0_inv, ml, mr, L, targetfreq, hasindel, FALSE, verbose);
    
    ChangeRateMatrix(ofp, Q_0K,   Q_0_inv,  K,  ml, mr, L, targetfreq, hasindel, TRUE,  verbose);
    ChangeRateMatrix(ofp, QR_0KR, QR_0_inv, KR, ml, mr, L, targetfreq, hasindel, FALSE, verbose);
 
    free(Q_0K);
    free(QR_0KR);
  }
  
   /* if this process involves indels, we have to evolve the marginal probabilities too
    */
  if (hasindel) {
    EvolveIndelMarginals(stdout, K,  Q_0,  mr, tfactor, L, FALSE, verbose);
    EvolveIndelMarginals(stdout, KR, QR_0, ml, tfactor, L, FALSE, verbose);
  }

  if (changefreq) {
    ChangeFrequencies(ofp, Q_0,  K,  mr, L, hasindel, verbose); /* Left  conditionals saturate to right marginals */
    ChangeFrequencies(ofp, QR_0, KR, ml, L, hasindel, verbose); /* Right conditionals saturate to left  marginals */
  }
  
  /* calculate Qp = Q_0 * exp{tfactor*K}
   */
  Condi_From_Rate(&Qp,  K,  Q_0,  tfactor, L, pedantic, verbose);
  Condi_From_Rate(&QRp, KR, QR_0, tfactor, L, pedantic, verbose);
  CopyMatrix(Q,  Qp,  L, L);
  CopyMatrix(QR, QRp, L, L);
  
  /* consistency check, rows should add up to one
   */	
  for (i = 0; i < L; i++)
    CheckSingleProb(Q+i*L, L);
  for (i = 0; i < L; i++)
    CheckSingleProb(QR+i*L, L);
  
  if (verbose) {
    fprintf(ofp, "Q(j|i, r*t_o=%f)  probabilities\n", tfactor);
    PrintProbs(ofp, Q, L);
    fprintf(ofp, "QR(j|i, r*t_o=%f)  probabilities\n", tfactor);
    PrintProbs(ofp, QR, L);
  }
  
  if (verbose) {
   check_Q_0_reversibility(Q_0, QR_0, ml, mr, L, hasindel);
   check_reversibility    (Q,   QR,   ml, mr, L);
  }
  
  
  free(S);
  free(K);
  free(Q_0_inv);
  free(Qp);
  free(QRp);
}

/* Function: RIBOConditionalsEvolved()
 * Date:     ER, Tue Oct  7 14:41:49 CDT 2003 [St. Louis]
 *
 * Purpose:  Given a set of conditionals probabilities Q(a|b, t) 
 *           at a given evolutionary time t, 
 *           and set of conditional probabilities at time zero Q_0
 *         
 *           calculate Q(a,b| r*t) using a Markov chain model for evolution
 *  
 * Method:   Q(a | b, r*t) = Q_0 * exp{r * log[Q_0^{-1} * Q}
 *
 *
 *
 *
 * Args:              
 *
 * Returns: Qr(LxL). 
 *          Qr is allocated here, freed by caller.
 *           
 */
void
RIBOConditionalsEvolved(FILE *ofp, fullcondmat_t *ribocond, condmatrix_t *ribocond_nought, 
			double tfactor, double *targetpair, int changepair, int pedantic, int verbose)
{
  double  *Q_0_inv;
  double  *Qp;
  double  *K;
  double  *S;
  double   tmax;
  double   xtime;
  int      L;
  int      L2;
  int      n_times;
  int      i;

  L  = ribocond->marg->size;
  L2 = ribocond->cond->size;

  tmax = TMAX;

  n_times = (int)tfactor / (int)tmax;
  xtime   = tfactor - tmax*(double)n_times;

  /* consistency check, rows should add up to one
   */	
  for (i = 0; i < L2; i++)  CheckSingleProb(ribocond->cond->matrix+i*L2, L2);
  
  if (verbose) {
    fprintf(ofp, "Q(j|i, t_*)  probabilities\n");
    PrintProbs(ofp, ribocond->cond->matrix, L2);
    fprintf(ofp, "Q_0(j|i)  probabilities\n");
    PrintProbs(ofp, ribocond_nought->matrix, L2);
    fprintf(ofp, "pair-Marginal  probabilities\n");
    PrintProbs(ofp, ribocond->marg->matrix, L);
  }
  
  /* calculate Q_0_inv
   */
  Q_0_inv = Cal_M_Inv(ofp, ribocond_nought->matrix, L2, verbose);
  
  /* construct K = log [Q_0_inv * Q]
   */
  Comp_M_N_Prod(ofp, Q_0_inv, ribocond->cond->matrix, L2, verbose); /*  multiply Q_0_inv*Q, dump it in Q */
  
  S = Cal_Id(L2);
  Comp_M_N_Sum(ofp, ribocond->cond->matrix, S, L2, L2, FALSE, verbose);   /* S = Q - Id  */
  
  /* check that Q is consistent with a markovian stationary 
   * model of evolution. 
   * In plain words it checks whether logQ is going to exist.
   *
   * Basically, it checks whether  all eigenvalues of Q - I
   * are between -1 and 1. 
   *
   * remember log (1+x) = x - x^2/2 + ...   converges for -1 < x <= 1
   *
   */
  
  if (verbose) isQconsistent(ofp, S, L2, pedantic, verbose);
  K = Cal_M_Taylor_Log(ofp, ribocond->cond->matrix, L2, verbose);
  
  if (verbose) {
    fprintf(ofp, "R  rate matrix\n");
    PrintProbs(ofp, K, L2);
  }
 
  if (pedantic)
    islogQconsistent(ofp, ribocond_nought->matrix, K, L2, L2, L2, verbose); /*  check that K is consistent with a 
						      *  markovian stationary model of evolution. 
						      *
						      *  Basically, it checks whether all entries of  
						      *  Q_0 + epsilon*Q_0*K are none negatives.
						      */

  /* If we are told to change the stationary frequencies we do it here.
   * This will also modify the rate matrix K, while keeping the exchangeability parameters S
   * defined as
   *            (Q_0 K)_ij  = S_ij (p_j/p_i)^{1/2}  i \neq j 
   *
   *   to  
   *            (Q_0 K')_ij = S_ij (p'_j/p'_i)^{1/2}
   *
   *    so       K' = Q_0_inv * (Q_0 K')
   *
   */
  if (changepair) 
    ChangeRIBORateMatrix(ofp, K, Q_0_inv, ribocond->marg->matrix, L, targetpair, verbose);
  
   /* calculate Qp = Q_0 * exp{tfactor*K}
   */
  Condi_From_Rate(&Qp, K, ribocond_nought->matrix, tfactor, L2, pedantic, verbose);
  CopyMatrix(ribocond->cond->matrix, Qp, L2, L2);
  
  /* consistency check, rows should add up to one
   */	
  for (i = 0; i < L2; i++)
    CheckSingleProb(ribocond_nought->matrix+i*L2, L2);
  
  DLog2(ribocond->cond->matrix, L2*L2);

  if (verbose) {
    fprintf(ofp, "Q(j|i, r*t_o=%f)  probabilities\n", tfactor);
    PrintFullRIBOCondMatrix(stdout, ribocond, TRUE, FALSE);
  }
  
  free(S);
  free(K);
  free(Q_0_inv);
  free(Qp);
}


/* Function: ConditionalsEvolved_2()
 * Date:     ER, Mon May  8 10:33:18 CDT 2000 [St. Louis]
 *
 * Purpose:  Given a set of conditionals probabilities Q(a|b, t) 
 *           at a given evolutionary time t, 
 *           and set of conditional probabilities at time zero Q_0
 *         
 *           calculate Q(a,b| r*t) 
 *           this is a more general model than the previous one:
 *  
 * Method:   Q(a | b, r*t) = Q_0 +  exp{ r * log[I + R^{-1}(Q-Q_0)] }
 *
 *           for once, the property Q(t+s) = Q(t)*Q(s) is not satisfied any more.
 *
 * Args:              
 *
 * Returns: Qr(LxL).  *          Qr is allocated here, freed by caller.
 *           
 */
void
ConditionalsEvolved_2(FILE *ofp, double *Q, double *Q_0, double *R, int L, double tfactor, double *targetfreq, 
		      int changefreq, int pedantic, int verbose)
{
  double  *R_inv;
  double  *Qr;
  double  *Qp;
  double  *K;
  double  *Id;
  double   tmax;
  double   xtime;
  int      n_times;
  int      i;

  tmax = TMAX;

  n_times = (int)tfactor / (int)tmax;
  xtime   = tfactor - tmax*(double)n_times;

  /* consistency check, rows should add up to one
   */	
  for (i = 0; i < L; i++)
    CheckSingleProb(Q+i*L, L);
  
  Id  = Cal_Id(L);
  
  /* calculate R_inv 
   */
  R_inv = Cal_M_Inv(ofp, R, L, verbose);
  
  /* calculate I + R_inv*(Q-Q_0)  dump it in Q 
   */
  Comp_M_N_Sum(ofp, Q, Q_0, L, L, FALSE, verbose);    /*  add:      Q - Q_0,   dump it in Q */
  CopyMatrix(Q, Q_0, L, L);
  Comp_M_N_Prod(ofp, R_inv, Q, L, verbose);           /*  multiply: R_inv * Q, dump it in Q */

  if (pedantic)
    isQconsistent(ofp, Q, L, pedantic, verbose);  /* check that Q is consistent with a markovian stationary 
						   * modelof evolution. 
						   * In plain words it checks whether log(I+Q) is going to exist.
						   *
						   * Basically, it checks whether  all eigenvalues of Q
						   * are between -1 and 1. 
						   *
						   * remember log (1+x) = x - x^2/2 + ...   converges for -1 < x <= 1
						   *
						   */

  Comp_M_N_Sum(ofp, Id, Q, L, L, TRUE, verbose);   /*  add:      Id + Q,    dump it in Q */

  /* construct K = log [I + R_inv*(Q-Q_0)]   */
  K = Cal_M_Taylor_Log(ofp, Q, L, FALSE);
  
  if (pedantic)
    islogQconsistent_2(ofp, Q_0, R, K, L,  L,  L, verbose); /*  check that K is consistent with a 
							     *  markovian stationary model of evolution. 
							     *
							     *  Basically, it checks whether all entries of  
							     *  Q_0 + epsilon*R*K are none negatives.
							     */
  
  /* calculate Qr = Q_0 + R * [ exp{tfactor*K} - Id ]
   */
  Qp = (double *)Cal_Id(L);
  while (n_times > 0) {
    Qr = Cal_M_Exp(ofp, K, L, tmax, verbose);
    Comp_M_N_Prod(ofp, Qr, Qp, L, FALSE); /*  multiply Qr^(n_times*tmax), dump it in Qp */
    n_times --;
    free (Qr);
  }
  Qr = Cal_M_Exp(ofp, K, L, xtime, verbose);
  Comp_M_N_Prod(ofp, Qr, Qp, L, FALSE); /*  multiply Qr^xtime, dump it in Qp */
  
  Comp_M_N_Prod(ofp, R, Qr, L, FALSE);               /* multiply: R*Qp,   dump it in Qp */
  Comp_M_N_Sum(ofp, Q_0, Qr, L, L, TRUE, FALSE);     /* add:      Q_0+Qp, dump it in Qp */
  
  if (verbose) {
    fprintf(ofp, "Q(j|i, r*t_o)  probabilities\n"); 
    PrintProbs(ofp, Qp, L);
  }
  
  CopyMatrix(Q, Qp, L, L);

  /* consistency check, rows should add up to one
   */	
  for (i = 0; i < L; i++)
    CheckSingleProb(Q+i*L, L);
  
  free(K);
  free(Id);
  free(R_inv);
  free(Qr);
  free(Qp);

}

/* Function: EvolveIndelMarginals()
 * Date:     ER, Wed Sep 17 14:35:07 CDT 2003 [St. Louis]
 *
 * Purpose:  Given       a LxL matrix P(i,j)   of Joint probabilities, 
 *                       a LxL matrix Q_0(i,j) of conditional probabilities at t=0
 *
 *                       a L vector of marginal probs of the form   pmar    = [p_i(1-lambda*), lamda*]
 *
 *           Use P and Q_0 to calculate the marginals at time infty p_infty = [p_i(1-lambda_infty), lamda_infty] 
 *
 *           evolve lambda(t) = lambda_infty[1-exp{t*log(1-lambda*\/lamda_infty)]
 *
 *           construct the evolved marginals as pmar(t) = [p_i(1-lambda(t)), lamda(t)]
 *
 *
 * Returns:  void.
 */
void
EvolveIndelMarginals(FILE *ofp, double *R, double *Q_0, double *pm, double tfactor, int L, int pedantic, int verbose)
{
  double *pinfty;
  double  lambda_star;
  double  lambda_infty;
  int     i;

  pinfty = (double *) MallocOrDie (sizeof(double) * L); 

   if (verbose) {
    fprintf(ofp, "Target marginals\n");
    PrintVectorProbs(ofp, pm, L);
  }

  /* saturation probabilitie
   */
  pinfty = SaturationProbs(R, Q_0, L, FALSE, verbose);
 
  /* evolve the sigle probs -- only evolve the last component
   */
  lambda_star  = pm[L-1];
  lambda_infty = pinfty[L-1];
  
  if (verbose) {
    fprintf(ofp, "INFTY-marginals\n");
    PrintVectorProbs(ofp, pinfty, L);
  }

  if (lambda_star > lambda_infty) Die ("EvolveIndelMarginals(): bad marginals");

  pm[L-1] = pinfty[L-1]*(1.0-EXP2(tfactor*LOG2(1.0-lambda_star/lambda_infty)));
  
  for (i = 0; i < L-1; i++) 
      pm[i] *= (1.0-pm[L-1]) /(1.0-lambda_star);
 
  CheckSingleProb(pm, L);
  
  if (verbose) {
    fprintf(ofp, "marginals evolved %f time\n", tfactor);
    PrintVectorProbs(ofp, pm, L);
    fprintf(ofp, "INFTY-marginals\n");
    PrintVectorProbs(ofp, pinfty, L);
  }

  free(pinfty);
}


/* Function: Joint_From_Condi()
 * Date:     ER, Mon Jun 12 14:41:47 CDT 2000 [St. Louis]
 *
 * Purpose:  Given       a LxL matrix Q(i,j) of conditional probabilities, 
 *           calculate the LxL matrix P(i,j) of joint       probabilities.
 *
 *           P(i,j) = Q(j|i) * pm[j] 
 *
 *           where P(i,j|t_o) are calculated according to 
 *           using a given pammodel t_o.
 *
 * Args:     Q  - LxL conditional prob matrix (prealloc)
 *           pm - L   single emision probabilities (prealloc)
 *
 * Returns:  P(LxL), conditional probabilities.
 *           P is alocated here, freed by caller.
 */
void
Joint_From_Condi(FILE *ofp, double *P, double *Q, double *pm, int L, int verbose)
{
  double *Qnewl;
  double *Qnewr;
  double *pml;
  double *pmr;
  double  norm = 0.0;
  int     i, j;
  
  if (verbose) {
    fprintf(ofp, "Q(i|j,t) probabilities\n");
    PrintProbs(ofp, Q, L);
    fprintf(ofp, "marginals probabilities\n");
    PrintVectorProbs(ofp, pm, L);
  }
  
  for (i = 0; i < L; i++) 
    for (j = 0; j < L; j++) {
      P[i*L+j] = Q[i*L+j] * pm[i]; 
      norm += P[i*L+j];
    }

  for (i = 0; i < L; i++) 
    for (j = 0; j < L; j++) 
      P[i*L+j] /= norm;
  
  if (verbose) { 
    fprintf(ofp, "P(i,j,t) probabilities\n");
    PrintProbs(ofp, P, L);

    CalculateConditionalsAndMarginals(ofp, P, &Qnewl, &Qnewr, &pml, &pmr, L, verbose);
    fprintf(ofp, "newQl(i|j,t) probabilities\n");
    PrintProbs(ofp, Qnewl, L);
    fprintf(ofp, "newQr(i|j,t) probabilities\n");
    PrintProbs(ofp, Qnewr, L);
    fprintf(ofp, "marginals probabilities\n");
    PrintVectorProbs(ofp, pml, L);
    PrintVectorProbs(ofp, pmr, L);
    
    free(Qnewl);
    free(Qnewr);
    free(pml);
    free(pmr);
  }
  
  /* consistency check
   */
  CheckSingleProb(P, L*L);
  
}
 
void
Joint_From_CondiR(FILE *ofp, double *P, double *QR, double *pr, int L, int verbose)
{
  double *Qnewl;
  double *Qnewr;
  double *pml;
  double *pmr;
  int     i, j;
  
  if (verbose) {
    fprintf(ofp, "QR(i|j,t) probabilities\n");
    PrintProbs(ofp, QR, L);
    fprintf(ofp, "Rmarginals probabilities\n");
    PrintVectorProbs(ofp, pr, L);
  }
  
  for (i = 0; i < L; i++) 
    for (j = 0; j < L; j++)
      P[i*L+j] = QR[j*L+i] * pr[j]; 
  
  if (verbose) { 
    fprintf(ofp, "P(i,j,t) probabilities\n");
    PrintProbs(ofp, P, L);

    CalculateConditionalsAndMarginals(ofp, P, &Qnewl, &Qnewr, &pml, &pmr, L, verbose);
    fprintf(ofp, "newQl(i|j,t) probabilities\n");
    PrintProbs(ofp, Qnewl, L);
    fprintf(ofp, "newQr(i|j,t) probabilities\n");
    PrintProbs(ofp, Qnewr, L);
    fprintf(ofp, "marginals probabilities\n");
    PrintVectorProbs(ofp, pml, L);
    PrintVectorProbs(ofp, pmr, L);
    
    free(Qnewl);
    free(Qnewr);
    free(pml);
    free(pmr);
  }
  
  /* consistency check
   */
  CheckSingleProb(P, L*L);
  
}
 
/* Function: Joint2Joint()
 * Date:     ER, Wed Apr 12 18:11:23 CDT 2000 [St. Louis]
 *
 * Purpose:  Given a set of joint probabilities P(a,b| t_o) 
 *           at a given evolutionary time t_o, 
 *           and set of conditional probabilities at time zer0 P^o
 *         
 *           calculate P(a,b| tfactor*t_o) using a Markov chain model for evolution
 *  
 * Method:   (1) calculate conditionals: P(a,b | t_o) --> Q(a | b,t_o) 
 *            
 *           (2) calculate marginals: P(a,b | t_o) --> pmar(a | b,t_o)  
 *            
 *           (3) evolve conditionals Q(a | b, r*t_o) = Q_0 * exp{ r * log[Q_0^{-1} * Q] }
 *
 *           (4) construct joints using same marginals Q(a | b, r*t_o), pmar(a | b,t_o)  --> P(a,b | r*t_o)
 *
 * Args:              
 *
 * Returns:  (void)
 *           
 */
void
Joint2Joint(double *p_joint, double *ql_nought, double *qr_nought, int L, double tfactor, double *targetfreq, 
	    int changefreq, int hasindel, int ispairprob, int pedantic, int verbose)
{
  double *p_margl;
  double *p_margr;
  double *p_condl;
  double *p_condr;
  

  /* calculate the matrix of conditional probabilities (Q) and marginals from joint probabilities (P)
   */
  CalculateConditionalsAndMarginals(stdout, p_joint, &p_condl, &p_condr, &p_margl, &p_margr, L, verbose);

  /* evolve conditionals, change the marginals if asked to do so
   */
  ConditionalsEvolved(stdout, p_condl, p_condr, ql_nought, qr_nought, p_margl, p_margr, L, tfactor, 
		      targetfreq, changefreq, hasindel, pedantic, verbose);
 
  /* Reconstruct the new evolved Joint probabilities
   */
  Joint_From_Condi(stdout, p_joint, p_condl, p_margl, L, verbose);
  
  free(p_margl);
  free(p_margr);
  free(p_condl);
  free(p_condr);
}


/* Function: Joint2JointGap()
 * Date:     ER, Tue Jan 14 11:24:14 CST 2003 [St. Louis]
 *
 * Purpose:  Given a set of joint probabilities P(a,b| t_*)
 *           at a given evolutionary time t_*,
 *           the set of conditional probabilities at time zero P^o,
 *           and the set of single probs at the time of interests pv
 *
 *           calculate P(a,b| tfactor*t_o) using a Markov chain model for evolution.
 *           For probabilities that include gaps.
 *
 * Method:   (1) calculate conditionals: P(a,b | t_*) --> Q(a | b,t_*)
 *
 *           (4) evolve conditionals Q(a | b, r*t_*) = Q_0 * exp{ r * log[Q_0^{-1} * Q] }
 *
 *           (6) construct joints with new probs Q(a | b, r*t_*), pv(a | b, r*t_*)  --> P(a,b | r*t_*)
 *
 * Args:
 *
 * Returns:  (void)
 *
 */
void
Joint2JointGap(double *p_joint, double *q_nought, double *pv, int L, double tfactor, double *targetfreq, 
	       int changefreq,  int pedantic, int verbose)
{
  double *p_cond;
  double *p_condr;
  double *q_noughtr;
  double *pvr;
  double  sum;
  double  alpha;


  /* calculate the matrix of conditional probabilities (Q) from joint probabilities (P)
   */
  p_cond  = Condi_From_Joint (stdout, p_joint, L, verbose);
  p_condr = CondiR_From_Joint(stdout, p_joint, L, verbose);

  q_noughtr = (double *) MallocOrDie (sizeof(double) * L * L);
  pvr       = (double *) MallocOrDie (sizeof(double) * L);
  CopyMatrix(q_noughtr, q_nought, L, L);
  CopyVector(pvr, pv, L);

  /* evolve conditionals
   */
  ConditionalsEvolved(stdout, p_cond, p_condr, q_nought, q_noughtr, pv, pvr, L, tfactor, targetfreq, changefreq, FALSE, pedantic, verbose);

  Joint_From_Condi(stdout, p_joint, p_cond, pv, L, verbose);

  CheckSingleProb(p_joint, L*L);

  if (verbose) {
    fprintf(stdout, "P_COND(i,j| r*t_o=%f)  Cond probabilities\n", tfactor);
    PrintProbs(stdout, p_cond, L);
    fprintf(stdout, "Psingle(i| r*t_o=%f) single nt probabilities\n", tfactor);
    PrintVectorProbs(stdout, pv, L);
    fprintf(stdout, "P_JOINT(i,j| r*t_o=%f)  Joint probabilities\n", tfactor);
    PrintProbs(stdout, p_joint, L);
 }

  free(pvr);
  free(p_cond);
  free(p_condr);
  free(q_noughtr);
}


/* Function: MarginalizeJointProbs()
 * Date:     ER, Wed Aug 21 13:19:04 CDT 2002 [St. Louis]
 *
 * Purpose:  Given a jp[idx(dim,dim)] joint probability distrubution, 
 *           marginalize to calculate sp[dim].
 *
 * Args:    jp - dimxdim A..UxA..U joint prob matrix (prealloc)
 *          sp - dim seqX emission prob (prealloc)
 *
 * Returns:  (void)
 *           Fills in sp (already allocated)
 */
void
MarginalizeJointProbs(double *jp, double *sp, int dim, int which)
{
  int x;
  int y;

  /* Zero null model
   */
  for (x = 0; x < dim; x++)
    sp[x] = 0.0;

  /* Marginalize and average over Y positions
   */
  for (x = 0; x < dim; x++)
    for (y = 0; y < dim; y++)
      if      (which == 0) sp[x] += jp[x*dim+y];
      else if (which == 1) sp[x] += jp[y*dim+x];
      else if (which == 2) sp[x] += 0.5*(jp[x*dim+y]+jp[y*dim+x]);
      else Die ("You have to marginalize X(=0) or Y (=1), or averaging (2) nothing else allowed\n");

  CheckSingleProb(sp, dim);
}

/* Function: RateFromConditionals()
 * Date:     ER, Fri Sep 12 10:09:25 CDT 2003 [St. Louis]
 *
 * Purpose:  Given a set of conditionals probabilities Q(a|b, t) 
 *           calculate the rate matrix
 *  
 *  
 * Method:   R = log[Q_0^{-1} * Q}
 *
 *
 * Args:              
 *
 * Returns: Qr(LxL). 
 *          Qr is allocated here, freed by caller.
 *           
 */
void
RateFromConditionals(FILE *ofp, double *R, double *Q, double *Q_0, int L, double tfactor, int pedantic, int verbose)
{
  double  *AUX;
  double  *Q_0_inv;
  double  *K;
  double  *S;
  int      i;

  /* consistency check, rows should add up to one
   */	
  for (i = 0; i < L; i++)
    CheckSingleProb(Q+i*L, L);
  
  if (verbose) {
    fprintf(ofp, "Q(j|i, %.2f t_*)  probabilities\n", tfactor);
    PrintProbs(ofp, Q, L);
  }
  /* calculate Q_0_inv
   */
  Q_0_inv = Cal_M_Inv(ofp, Q_0, L, verbose);
  if (verbose) {
    fprintf(ofp, "Q_0_inv(j|i)  probabilities\n");
    PrintProbs(ofp, Q_0_inv, L);
  }
  
  /* construct K = log [Q_0_inv * Q]
   */
  AUX = Cal_M_N_Prod(ofp, Q_0_inv, Q, L, L, L, verbose); /*  multiply Q_0_inv*Q, dump it in AUX */
  
  S = Cal_Id(L);
  Comp_M_N_Sum(ofp, AUX, S, L, L, FALSE, verbose);   /* S = AUX - Id  */
 
  if (verbose) 
    isQconsistent(ofp, S, L, pedantic, verbose);  /* check that Q is consistent with a markovian stationary 
						   * model of evolution. 
						   * In plain words it checks whether logQ is going to exist.
						   *
						   * Basically, it checks whether  all eigenvalues of Q - I
						   * are between -1 and 1. 
						   *
						   * remember log (1+x) = x - x^2/2 + ...   converges for -1 < x <= 1
						   *
						   */
  K = Cal_M_Taylor_Log(ofp, AUX, L, verbose);
  MultiplyMatrix ((1.0/tfactor), L, K);

  if (verbose) {
    fprintf(ofp, "R  rate matrix\n");
    PrintProbs(ofp, K, L);
  }
 
  if (pedantic)
    islogQconsistent(ofp, Q_0, K, L, L, L, verbose); /*  check that K is consistent with a 
						      *  markovian stationary model of evolution. 
						      *
						      *  Basically, it checks whether all entries of  
						      *  Q_0 + epsilon*Q_0*K are none negatives.
						      */
  
  CopyMatrix(R, K, L, L);

  free(K);
  free(S);
  free(AUX);
  free(Q_0_inv);
}

/* Function: SaturationProbs()
 * Date:     ER, Thu Sep 11 16:39:06 CDT 2003 [St. Louis]
 *
 * Purpose:  Given a set of joint P(a,b| t_o) 
 *           at a given evolutionary time t_o, 
 *           and set of conditional probabilities at time zer0 P^o
 *         
 *           calculate P(a,b| tfactor*t_o) using a Markov chain model for evolution
 *  
 * Method:   (1) calculate conditionals: P(a,b | t_o) --> Q(a | b,t_o) 
 *            
 *           (3) evolve conditionals at time infity Q(a | b, r*t_o) = Q_0 * exp{ r * log[Q_0^{-1} * Q] }
 *
 *           (4) extract the saturation probs.
 *
 * Args:              
 *
 * Returns:  saturation probabilities
 *           
 */
double *
SaturationProbs(double *R, double *Q_0, int L, int pedantic, int verbose)
{
  double *psat;
  double *Q;
  double  tfactor = 500.0;
  double  margin = 0.0005;
  int     i, j;
  
  psat = (double *) MallocOrDie (sizeof(double) * L);

  /* calculate Q^\infty = Q_0 * exp{tfactor*K}
   */
  Condi_From_Rate(&Q, R, Q_0, tfactor, L, pedantic, verbose);

  for (j = 0; j < L; j++) {
    psat[j] = Q[j*L+j];
    for (i = 1; i < L; i++)
      if (psat[j]-Q[i*L+j]>margin) Die ("SaturationProbs(): you have not reached saturation yet\n");
  }
  
  CheckSingleProb(psat, L);
 
  free (Q);

  return psat;
}


/* Function: TimeIdCorrelation()
 *
 * Date:     ER, Thu Nov 14 14:16:12 CST 2002 [St. Louis]
 *
 *
 *     id = 100 - div.a * time ^ div.b ==>  log (time) = 1/div.b log[(100-id)/div.a]
 *
 */
double 
TimeIdCorrelation(struct divergence_s div, double id)
{
  double time;
  double id_cutoff = 0.0;

  if (div.a <=   0.) { time = 1.0; return time; }
  if (id    == 100.) { time = 0.0; return time; }

  if (id >= id_cutoff)
    time = LOG2( (100.0-id) / div.a);
  else
    time = LOG2( (100.0-id_cutoff) / div.a);

  if (div.b != 0.0) 
    time /= div.b;
  else  
    Die ("TimeIdCorrelation() error. Division by zero\n");

  time = EXP2(time);

  return time;
}

/* Function: TransitionsEvolvedLinear()
 * Date:     ER, Fri Jul 26 15:20:09 CDT 2002 [St. Louis]
 *
 * Purpose:  Given a set of conditionals probabilities Q(a|b, t) 
 *           at a given evolutionary time t, 
 *           and set of conditional probabilities at time zero Q_0
 *         
 *           calculate Q(a,b| r*t) using a Markov chain model for evolution
 *  
 * Method:   
 *
 *                    q^i(t) = q^i_0 + r^i [ exp{tA^i} - I ]  ---   A(nxn), q(1xn), q_0(1xn), r(1xn)
 *
 *
 *              | q^1 |                           | q^1_o |                     |  0  |                           | 1 |
 *              |  .  |                           |   .   |                     |  0  |                           | 1 |
 *              |  .  |                           |   .   |                     |  0  |                           | 1 |
 *              |  .  |                           |   .   |                     |  0  |                           | 1 |
 *        Q^* = | q^i |                     Q_0 = | q^i_o |               R^i = |  0  |          A^i = 1/t* log [ | 1 | (q^i - q^I_o) + I ]
 *              |  .  |                           |   .   |                     | r^i |                           | 1 |
 *              |  .  |                           |   .   |                     |  0  |                           | 1 |
 *              |  .  |                           |   .   |                     |  0  |                           | 1 |
 *              | q^n |                           | q^n_o |                     |  0  |                           | 1 |
 *
 *
 *
 *     THEN,      Q(t)  =  Q_0  + Sum_i R^i [ exp(tA^i) - I ]
 *
 * Args:              
 *
 * Returns: Qr(LxL). 
 *          Qr is allocated here, freed by caller.
 *           
 */
void
TransitionsEvolvedLinear(FILE *ofp, double *q, double *q_zero, double *r, double *r_diag, double *r_fix, 
			 int L, double tfactor, int pedantic, int verbose)
{
  double  *Id;        /* LxL array */
  double  *A;         /* LxL array */
  double  *K;         /* LxL array */
  double  *Qr;        /* 1xL array */
  int      i, j;

  Id = Cal_Id(L);

  /* Allocate 
   */
  A = (double *) MallocOrDie (sizeof(double) * L * L);

  /* 
   *
   *                       | 1 |           | K1                |
   *                       | 1 |           |    K2             |
   *                       | 1 |           |       K3          |
   *    Calculate  A = I + | 1 | r_fix +   |          .        | 
   *                       | 1 |           |            .      | 
   *                       | 1 |           |              .    | 
   *                       | 1 |           |                Kn | 
   *
   *
   *          with Ki = r_diag^i =  ( q^i - q^i_zero - beta * r^i_fix) / r^i
   *
   */

  for (i = 0; i < L; i++)
    for (j = 0; j < L; j++) {
      A[i*L+j] = r_fix[j];  

      if (i==j) A[i*L+j] += r_diag[i];
    }
  
  if (pedantic)
    isQconsistent(ofp, A, L, pedantic, verbose);  /* check that A is consistent with a markovian stationary 
						   * model of evolution. 
						   * In plain words it checks whether log(I+A) is going to exist.
						   *
						   * Basically, it checks whether  all eigenvalues of A
						   * are between -1 and 1. 
						   *
						   * remember log (1+x) = x - x^2/2 + ...   converges for -1 < x <= 1
						   *
						   */

  /* Now add the identity to complete the 
   * definition of A
   */
  for (i = 0; i < L; i++) A[i*L+i] += 1.0;

  if (FALSE) {
    fprintf(ofp, "\n EXP[ t^* A ] \n");

    for (i = 0; i < L; i++) {
      for (j = 0; j < L; j++) 
        fprintf(ofp, "%.4f ", A[i*L+j]);
      fprintf(ofp, "\n");
    }
    fprintf(ofp, "\n\n");
  }

  /* construct K = log [A]
   */
  K = Cal_M_Taylor_Log(ofp, A, L, verbose);
  
  if (pedantic)
    islogQconsistent_2(ofp, q_zero, r, K, 1, L, L, verbose); /*  check that K is consistent with a 
							      *  markovian stationary model of evolution. 
							      *
							      *  Basically, it checks whether all entries of  
							      *  q_zero + epsilon*q_zero*K are none negatives.
							      */
  
  /* calculate Qr = Q_0 + R * [ exp{tfactor*K} - Id]
   */    
  Comp_M_Exp(ofp, K, L, tfactor, verbose);                /* exp:      exp(t*K),  dump it in K      */
  Comp_M_N_Sum(ofp, K, Id, L, L, FALSE, verbose);             /* add:      K-Id,      dump it in K      */
  CopyMatrix(K, Id, L, L);
  Qr = Cal_M_N_Prod(ofp, r, K, 1, L, L, verbose);             /* multiply: r*K,       dump it in Qr     */
  Comp_M_N_Sum(ofp, q_zero, Qr, 1, L, TRUE, verbose);         /* add:      q_zero+Qr, dump it in Qr     */
  
  
  if (verbose) {
    fprintf(ofp, "\nT(i->j | t)  Transition probabilities at time %.3f\n", tfactor);
    for (i = 0; i < L; i++) 
        fprintf(ofp, "%.4f ", Qr[i]);
    fprintf(ofp, "\n----------------------------------------------------------\n");
  }
  
  CopyMatrix(q, Qr, 1, L);
  /* consistency check, rows should add up to one
   */	
  CheckSingleProb(q, L);
  
  free(K);
  free(A);
  free(Id);
  free(Qr);

}

/* Function: TransitionsEvolved()
 * Date:     ER, Tue Aug 13 16:17:27 CDT 2002 [St. Louis]
 *
 * Purpose:  Given a set of conditionals probabilities Q(a|b, t) 
 *           at a given evolutionary time t, 
 *           and set of conditional probabilities at time zero Q_0
 *         
 *           calculate Q(a,b| r*t) using a Markov chain model for evolution
 *  
 * Method:   
 *
 *                    q^i(t) = q^i_0 + r^i [ exp{tA^i} - I ]  ---   A(nxn), q(1xn), q_0(1xn), r(1xn)
 *
 *
 *              | q^1 |                           | q^1_o |                     |  0  |                           | 1 |
 *              |  .  |                           |   .   |                     |  0  |                           | 1 |
 *              |  .  |                           |   .   |                     |  0  |                           | 1 |
 *              |  .  |                           |   .   |                     |  0  |                           | 1 |
 *        Q^* = | q^i |                     Q_0 = | q^i_o |               R^i = |  0  |          A^i = 1/t* log [ | 1 | r^fix + Diag(e^(-K^1)) ]
 *              |  .  |                           |   .   |                     | r^i |                           | 1 |
 *              |  .  |                           |   .   |                     |  0  |                           | 1 |
 *              |  .  |                           |   .   |                     |  0  |                           | 1 |
 *              | q^n |                           | q^n_o |                     |  0  |                           | 1 |
 *
 *
 *
 *     THEN,      Q(t)  =  Q_0  + Sum_i R^i [ exp(tA^i) - I ]
 *
 * Args:              
 *
 * Returns: Qr(LxL). 
 *          Qr is allocated here, freed by caller.
 *           
 */
void
TransitionsEvolved(FILE *ofp, double *q, double *q_zero, double *q_infty, double *r_diag, 
		   int L, double tfactor, int pedantic, int verbose)
{
  double  *Id;        /* LxL array */
  double  *A;         /* LxL array */
  double  *K;         /* LxL array */
  double  *Q;         /* 1xL array */
  double  *r;         /* 1xL array */
  double   w = 0.0;
  int      isdiagonal = 1;
  int      i, j;

  Id = Cal_Id(L);

  if (verbose) {
    fprintf(ofp, "\nT(i->j | t)  Transition probabilities at time zero\n");
    for (i = 0; i < L; i++) 
      fprintf(ofp, "%.4f ", q_zero[i]);
    fprintf(ofp, "\nT(i->j | t)  Transition probabilities at time star\n");
    for (i = 0; i < L; i++) 
      fprintf(ofp, "%.4f ", q[i]);
    fprintf(ofp, "\nT(i->j | t)  Transition probabilities at time infty\n");
    for (i = 0; i < L; i++) 
      fprintf(ofp, "%.4f ", q_infty[i]);
    printf("\n");
  }
  
  /* Allocate 
   */
  A = (double *) MallocOrDie (sizeof(double) * L * L);

  /* 
   *
   *                      | e^(-K^1)                            |
   *                      |         e^(-K2)                     |
   *                      |                e^(-K3)              |
   *    Calculate  A =    |                       .             | 
   *                      |                          .          | 
   *                      |                             .       | 
   *                      |                             e^(-Kn) | 
   *
   *
   *          with   e^(-K^i) = 1 + r_diag^i =  1 + ( q^i - q^i_zero) / r^i
   *
   */

  for (i = 0; i < L; i++) A[i*L+i] = 1.0 + r_diag[i];

  if (verbose) {
    fprintf(ofp, "\n EXP[ t^* A ] \n");

    for (i = 0; i < L; i++) {
      for (j = 0; j < L; j++) 
        fprintf(ofp, "%.4f ", A[i*L+j]);
      fprintf(ofp, "\n");
    }
    fprintf(ofp, "\n\n");
  }


  /* Calculate exp (t A), 
   *
   *   If A is diagonal it is very simple:  exp(tA) =  exp[ t^* A ] * t
   *
   *   Otherwise have to calculate the log and then exponentiate.
   *
   */
  if (isdiagonal) { 
    K = (double *) MallocOrDie (sizeof(double) * L * L);
    for (i = 0; i < L; i++) 
      for (j = 0; j < L; j++) 
	if (i==j) {
	  if (tfactor > 0.0) K[i*L+j] = exp(log(A[i*L+j]) * tfactor); 
	  else               K[i*L+j] = 1.0;
	}
	else
	  K[i*L+j] = 0.0;
  } 
  
  else {
    /* construct K = log [A]
     */
    K = Cal_M_Taylor_Log(ofp, A, L, verbose);
    
    if (pedantic)
      islogQconsistent_2(ofp, q_zero, r, K, 1, L, L, verbose); /*  check that K is consistent with a 
								*  markovian stationary model of evolution. 
								*
								*  Basically, it checks whether all entries of  
							        *  q_zero + epsilon*q_zero*K are none negatives.
							        */
    
    /* exp{tfactor*K} 
     */    
    Comp_M_Exp(ofp, K, L, tfactor, verbose);                /* exp:      exp(t*K),  dump it in K      */
  }
  

  /* Calculate w(t) = u^T exp(tA) r
   */
  r = Cal_M_N_Sum(ofp, q_zero, q_infty, 1, L, FALSE, verbose);         /* add:      r = q_0 - q_infty,           */  

  for (i = 0; i < L; i++) 
    for (j = 0; j < L; j++)  
      w += K[i*L+j] * r[j];
 
  /* Calculate 
   *
   *           q(t) = q_0 + [ exp(tA) - I ] r
   *
   */
  Comp_M_N_Sum(ofp, K, Id, L, L, FALSE, verbose);                   /* add:      K-Id,      dump it in K      */
  CopyMatrix (K, Id, L, L);
  Q = Cal_M_N_Prod(ofp, r, K, 1, L, L, verbose);                    /* multiply: r*K,       dump it in q      */
  Comp_M_N_Sum(ofp, q_zero, Q, 1, L, TRUE, verbose);                /* add:      q_zero+q,  dump it in q      */
  
  
  /* Normalize the probabilities
   */
  for (i = 0; i < L; i++) 
    Q[i] /= (1. + w);

  if (verbose) {
    fprintf(ofp, "\nT(i->j | t)  Transition probabilities at time %.3f\n", tfactor);
    for (i = 0; i < L; i++) 
      fprintf(ofp, "%.4f ", Q[i]);
    fprintf(ofp, "\n----------------------------------------------------------\n");
  }
  
  CopyMatrix(q, Q, 1, L);

  /* consistency check, rows should add up to one
   */	
  CheckSingleProb(q, L);
  
  free(r);
  free(K);
  free(A);
  free(Id);
  free(Q);

 }


/* Function: TransitionsDirectoryCosines()
 * Date:     ER, Tue Aug 13 17:01:49 CDT 2002 [St. Louis]
 *
 * Purpose:  Fill R_diag ant T_rate of the transfer matrix for the OTH model
 *
 * Args:     T
 *           T_zero
 *
 *
 * Returns:  void. Calculates R_diag and T_rate. Both allocated here, freed by caller.
 */
void
TransitionsDirectoryCosines(double *q, double *q_zero, double *q_rate, double *R_diag, int dim)
{
  double *q_infty;
  double  sum = 0.0;
  double  angle;
  double  shift;
  int     num = 0;
  int     col;

  /* Allocate q^infty */
  q_infty = (double  *) MallocOrDie (sizeof(double) * dim);

  /* fill R_diag.  
   * 
   *  R_diag_i = cos(theta)
   * 
   *  cos (theta) = (q_i - q^0_i) / (q^0_i - q^infty_i) 
   * 
   * 
   *  -1 < cos(theta) <= 0 ===> pi/2 <= theta < pi --- 
   * 
   *  also because 0 <= q_0 - r <= 1  ===>  cos(theta) <= (q^* / q^0) - 1
   * 
   * 
   */
  angle = 3.0*PI/4.0;
  shift = 0.05;

  for (col = 0; col < dim; col++) {
    if (q[col] - q_zero[col] < MARGIN && q[col] - q_zero[col] > -MARGIN) { 
      R_diag[col] = 0.0;           
      sum += q_zero[col]; 
      num ++; 
    }
    else {
      if (cos(angle) <= q[col]/q_zero[col] - 1.0) R_diag[col] = cos(angle);
      else                                        R_diag[col] = q[col]/q_zero[col] - 1.0;
    }
  }
  
  /* fill T_rate.  
   * 
   * q^rate = q^infty - q^0
   * 
   * q^rate = (q_i - q^0_i) / cos (theta)
   * 
   *  If  cos (theta) = 0, q^rate is free with the condition    0 <= q^0 - q^rate <= 1
   * 
   */
  sum /= num;
  for (col = 0; col < dim; col++) {
    if (q[col] - q_zero[col] < MARGIN && q[col] - q_zero[col] > -MARGIN) 
      q_rate[col] = q_zero[col] - sum; 
    else {                                                                
      if (cos(angle) <= q[col]/q_zero[col] - 1.0) q_rate[col] = (q[col] - q_zero[col]) / R_diag[col];
      else                                        q_rate[col] = q_zero[col];
    }

    q_infty[col] = q_zero[col] - q_rate[col];
  }
 
  if (FALSE) {
    printf("RA_diag - OTH transfer matrix\n");
    for (col = 0; col < dim; col++) 
      printf("%f ", R_diag[col]);
    
    printf("\n");
    
    printf("TA_rate - OTH transfer matrix\n");
    for (col = 0; col < dim; col++) 
      printf("%f ", q_rate[col]);
    
    printf("\n");
    
    
    printf("TA_infty - OTH transfer matrix\n");
    for (col = 0; col < dim; col++) 
      printf("%f ", q_infty[col]);
    
    printf("\n");
  }

  /* paranoia - check that q^infty = q^0 + q_rate add up to one
   */
  CheckSingleProb(q_infty, dim);
    
  free(q_infty);
}

/* Function: TransitionsExpVector()
 * Date:     ER, Wed Aug 14 17:48:05 CDT 2002 [St. Louis]
 *
 * Purpose:  Fill R_diag vector and r vector
 *                 
 *               from the expressions:        r_i = q^0_i - q^infty_i
 *
 *                                       R^diag_i = ( q_i - q^0_i ) / r_i
 *                 
 * Args:     
 *
 * Returns:  void
 */
double *
TransitionsExpVector(double *T, double *T_zero, double *T_infty, int dim)
{
  double  *R_exp;
  double   sum   = 0.;
  int      col;
 
  /* check the three input probability vectors are in good order
   *
   * to have things well behaving one of these two conditions have to occur
   *
   *
   *            T^0_i < T^*_i < T^infty_i    or     T^infty_i < T^*_i < T^0_i  
   *
   */
  /*check_three_probs (T, T_zero, T_infty, dim);*/

  /* allocate R_diag
   */
  R_exp = (double *) MallocOrDie (sizeof(double) * dim);

  /* initialize
   */
  for (col = 0; col < dim; col++)
    R_exp[col] = 0.0;
  
  /* Calculate 
   *
   *    
   *
   */
  for (col = 0; col < dim; col++)  
    if (T_zero[col] - T_infty[col] < MARGIN && T_zero[col] - T_infty[col] > -MARGIN) 
      R_exp[col] = 0.0;     
    else
      R_exp[col] = (T[col] - T_zero[col]) / (T_zero[col] - T_infty[col]); 
  
  if (FALSE) {
    printf("R^exp - matrix\n");
    for (col = 0; col < dim; col++) 
      printf("%.4f ", R_exp[col]);
    printf("\n");
  }
  
  /* paranoia: 
   *
   *           Define the scalar product K.R = sum_k K[k] * R[k]
   *        
   *           IS R^diag.R = 0 ????
   *
   *           Vectors R and R^diag being orthogonals  guarantees that q(t) add up to one.
   *
   */
  for (col = 0; col < dim; col++)
     sum += (T_zero[col] - T_infty[col]) * R_exp[col];
  if (sum < -MARGIN && sum > MARGIN) Die ("Bad Diag Matrix, sum_col K[col] R[col] = %f", sum);
    
  return R_exp;
}

/* Function: TransitionsDiagVector()
 * Date:     ER, Fri Aug  9 15:43:08 CDT 2002 [St. Louis]
 *
 * Purpose:  Fill R_diag matrix 
 *                 
 *               from the expression: R^diag_i * r_i = q_i - q^0_i - (sum_l r_l) R^fix_i  
 *                 
 *                 
 *           R^diag_k = (q_k - q^0_k - R^fix_k ) / r_k
 *
 * Args:     R_diag -- transfer matrix for the trnasitions  of the model
 *
 * Returns:  R_diag is allocated and filled here. R_diag freed by caller.
 */
double *
TransitionsDiagVector(double *T, double *T_zero, double *T_rate, double *R_fix, int dim)
{
  double *R_diag;
  double  beta  = 0.;
  double  sum   = 0.;
  int     col;

  /* allocate R_diag
   */
  R_diag = (double *) MallocOrDie (sizeof(double) * dim);

  /* initialize
   */
  for (col = 0; col < dim; col++)
    R_diag[col] = 0.0;
  
  /* Calculate beta = sum_k R[k]
   *
   */
  for (col = 0; col < dim; col++) beta += T_rate[col];
  if (beta < -MARGIN || beta > MARGIN) Die ("Bad Rate Matrix in TransitionsDiagVector(), sum_col R[col] = %f", beta);
  
  /* Calculate R^diag
   *
   *                       R^diag_k = (q_k - q^0_k - R^fix_k ) / r_k
   *
   */
  for (col = 0; col < dim; col++) 
    R_diag[col] = (T[col] - T_zero[col] - R_fix[col]) / T_rate[col]; 
  
  if (FALSE) {
    printf("R^diag - matrix\n");
    for (col = 0; col < dim; col++) 
      printf("%.4f ", R_diag[col]);
    printf("\n");
  }
  
  /* paranoia: 
   *
   *           Define the scalar product K.R = sum_k K[k] * R[k]
   *        
   *           IS R^diag.R = 0 ????
   *
   *           Vectors R and R^diag being orthogonals  guarantees that q(t) add up to one.
   *
   */
  for (col = 0; col < dim; col++)
     sum += T_rate[col] * R_diag[col];
  if (sum < -MARGIN && sum > MARGIN) Die ("Bad Diag Matrix, sum_col K[col] R[col] = %f", sum);
    
  return R_diag;
}


/* Function: TransitionsFixVector()
 * Date:     ER, Fri Aug  9 15:32:29 CDT 2002 [St. Louis]
 *
 * Purpose:  Fill R_fix matrix 
 *                 
 *               from the expression: R^diag_i * r_i = q_i - q^0_i - (sum_l r_l) R^fix_i  
 *                 
 *                 
 *           R^fix_k = (q_k - q^0_k) / beta - R^diag_k * r_k
 *
 *           where  beta = (sum_l r_l) 
 *
 * Args:     R_fix -- transfer matrix for the trnasitions  of the model
 *
 * Returns:  R_fix is allocated and filled here. R_fix freed by caller.
 */
double *
TransitionsFixVector(double *T, double *T_zero, double *T_rate, double *R_diag, int dim)
{
  double *R_fix;
  double  beta  = 0.;
  double  sum   = 0.;
  int     col;
  
  /* allocate R_fix
   */
  R_fix = (double *) MallocOrDie (sizeof(double) * dim);
  
  /* initialize
   */
  for (col = 0; col < dim; col++)
    R_fix[col] = 0.0;
  
  /* Calculate beta = sum_k R[k]
   *
   */
  for (col = 0; col < dim; col++) beta += T_rate[col];
  if (beta < -MARGIN && beta > MARGIN) Die ("Bad Rate Matrix in TransitionsFixVector(), sum_col R[col] = %f", beta);
  
  /* Calculate R^fix
   *
   *        R^fix_k = (q_k - q^0_k) / beta - R^diag_k * r_k
   *  
   */
  for (col = 0; col < dim; col++) 
    R_fix[col] = (T[col] - T_zero[col]) / beta - R_diag[col] * T_rate[col]; 
  
  if (FALSE) {
    printf("R^fix - matrix\n");
    for (col = 0; col < dim; col++) 
      printf("%.4f ", R_fix[col]);
    printf("\n");
  }
  
  /* paranoia: 
   *            elements of R^fix should add up to zero
   *
   */
  for (col = 0; col < dim; col++)
    sum += R_fix[col];
  if (sum < -MARGIN || sum > MARGIN) Die ("Bad Fix Matrix, sum_col R_fix[col] = %f", sum);
  
  return R_fix;
}


/* Function: TransitionsRateVector()
 * Date:     ER, Wed Aug  7 09:52:32 CDT 2002 [St. Louis]
 *
 * Purpose:  Fill R the rate matrix 
 *                 
 *               from the expression: R^diag_i * r_i = q_i - q^0_i - (sum_l r_l) R^fix_i  
 *                 
 *                 
 *           r_k = (q_k - q^0_k) / R^diag_k - beta * R^fix_k / R^diag_k
 *
 *           where  beta = (sum_l r_l) = [ sum_l (q_l - q^0_l) / R^diag_l ] / 1 + sum_l R^fix_k / R^diag_k
 *
 * Args:     R -- transfer matrix for the trnasitions  of the model
 *
 * Returns:  R is allocated and filled here. R freed by caller.
 */
double *
TransitionsRateVector(double *T, double *T_zero, double *R_fix, double *R_diag, int dim)
{
  double *R;
  double  beta  = 0.;
  double  sum_q = 0.;
  double  sum_i = 0.;
  double  sum   = 0.;
  int     col;

  /* allocate R 
   */
  R = (double *) MallocOrDie (sizeof(double) * dim);
  
  /* initialize
   */
  for (col = 0; col < dim; col++)
    R[col] = 0.0;
  
  /* Calculate beta = sum_k R[k]
   *
   */
  for (col = 0; col < dim; col++) 
    if (R_diag[col] > MARGIN || R_diag[col] < -MARGIN) {
      sum_q += (T[col] -  T_zero[col]) / R_diag[col]; 
      sum_i += R_fix[col]              / R_diag[col];
    }
  
  if (sum_q != 0.) {
    if (1.0 + sum_i < MARGIN && 1.0 + sum_i > -MARGIN) 
      Die ("There is an inconsistency in your Rate matrix. Revise your directory cosines. sum_q = %f   1+sum_i = %f", sum_q, 1.0+sum_i);
    else
      beta += sum_q / (1.0 + sum_i);
  }
  
  /* fill R 
   *           R_k = ( q_k - q^0_k - beta R^fix_k ) / R^diag_k
   *  
   */
  for (col = 0; col < dim; col++) 
    if (R_diag[col] > MARGIN || R_diag[col] < -MARGIN) 
      R[col] += ( T[col] -  T_zero[col] - beta * R_fix[col] ) / R_diag[col]; 
  
  if (FALSE) {
    printf("R - rate matrix -  beta = %f\n", beta);
    for (col = 0; col < dim; col++) 
      printf("%.4f ", R[col]);
    printf("\n");
  }
  
  /* paranoia: 
   *
   *           Define the scalar product K.R = sum_k K[k] * R[k]
   *        
   *           IS R^diag.R = 0 ????
   *
   *           Vectors R and R^diag being orthogonals  guarantees that q(t) add up to one.
   *
   */
  for (col = 0; col < dim; col++)
    sum += R[col] * R_diag[col];
  if (sum < -MARGIN || sum > MARGIN) Die ("Bad Rate Matrix, sum_col K[col] R[col] = %f", sum);
  
  return R;
}

/* Function: check_three_probs()
 * Date:     ER, Wed Aug 14 18:12:30 CDT 2002 [St. Louis]
 *
 * Purpose:  check the three input probability vectors are in good order
 *
 *             to have things well behaving one of these two conditions have to occur
 *
 *
 *              T^0_i < T^*_i < T^infty_i    or     T^infty_i < T^*_i < T^0_i  
 *
 */
void
check_three_probs (double *T, double *T_zero, double *T_infty, int dim)
{
  int i;

  /* check for one of the two conditions
   *
   *   q_0 < q_* < q_infty    or  q_0 > q_* > q_infty
   */
  for (i = 0; i < dim; i++) 
    if      (T_infty[i] <= T[i] && T[i] <= T_zero[i]) ;
    else if (T_infty[i] >= T[i] && T[i] >= T_zero[i]) ;
    else Die ("check your parameters for state %d. Time zero:(%f), star:(%f) and infty:(%f) \n", i, T_zero[i], T[i], T_infty[i]);
  
  
  /*
   * Also check that if two of the probs are equal, the third it is also equal.
   */
  for (i = 0; i < dim; i++) {
    if ( T_zero[i] - T[i]  < MARGIN*MARGIN && T_zero[i]  - T[i] > -MARGIN*MARGIN  &&
	 (T_infty[i] - T[i] > MARGIN*MARGIN || T_infty[i] - T[i] < -MARGIN*MARGIN)  )
      Die ("check your parameters for state %d. Time zero:(%f), star:(%f) and infty:(%f) \n", i, T_zero[i], T[i], T_infty[i]);
  }
}


/* Function: isQconsistent()
 * Date:     ER, Thu Apr 20 14:19:42 CDT 2000 [St. Louis]
 *
 * Purpose:  Q(t) ---a matrix of conditional probabilities---
 *           is consistent with being derived from a markovian stationary
 *           model of evolution from:
 *                    Q_0 - a matrix of conditional probabilities at time zero
 *                    K - a matrix of rate of evolution
 *
 *           as
 *                     Q(t) = Q_0 * exp {t K}
 *
 *
 *           IF: Q_0 + epsilon * Q_0*K  can play the role of an "instant matrix of evolution"
 *
 *               that is, if all the entries are positive.
 *
 * Args:      Q_0 - LxL matrix of conditional prob matrix at time zero (prealloc)
 *           logQ - LxL log matrix of conditional prob matrix (prealloc)
 *
 * Returns:  1 if consistent, 0 otherwise.
 */
int
isQconsistent(FILE *ofp, double *Q, int L, int pedantic, int verbose)
{
  struct eigenvalue_s *eigen;
  double              *H;
  double               norm;
  double               isconsistent = TRUE;
  int                  i;
  
  H = HessenbergForm(ofp, Q, L, verbose);
  Hessenberg2Eigenvalues(ofp, H, L, &eigen, pedantic, verbose);

  if (verbose) {
      fprintf(ofp, "Q matrix\n"); 
      PrintProbs(ofp, Q, L);

      fprintf(ofp, "Hessenberg form\n"); 
      PrintProbs(ofp, H, L);
      
      fprintf(ofp, "Eigenvalues for Q\n"); 
      for (i = 0; i < L; i++) 
	fprintf(ofp, "%.4f %.4f \n", eigen->real[i], eigen->imag[i]);
  }
  
  /* The condition is that ALL the eigenvalues of Q are of norm smaller
   * than one. 
   */
  for (i = 0; i < L; i++) {
    norm = sqrt(eigen->real[i]*eigen->real[i]+eigen->imag[i]*eigen->imag[i]);
    if (norm > 1.0+MARGIN2) {
	Warn ("Q not consistent with a markovian stationary model of evolution.\n Norm of eigenvalue (%f, %f) is = %f", 
	      eigen->real[i], eigen->imag[i], norm);
	isconsistent = FALSE;
    }
}

  free(H);
  free(eigen->real);
  free(eigen->imag);
  free(eigen);

  return isconsistent;
}
  
/* Function: islogQconsistent()
 * Date:     ER, Fri Apr 14 12:06:24 CDT 2000 [St. Louis]
 *
 * Purpose:  Q(t) ---a matrix of conditional probabilities---
 *           is consistent with being derived from a markovian stationary
 *           model of evolution from:
 *                    Q_0 - a matrix of conditional probabilities at time zero
 *                    K - a matrix of rate of evolution
 *
 *           as
 *                     Q(t) = Q_0 * exp {t K}
 *
 *
 *           IF: Q_0 + epsilon * Q_0*K  can play the role of an "instant matrix of evolution"
 *
 *               that is, if all the entries are positive.
 *
 * Args:      Q_0 - LxL matrix of conditional prob matrix at time zero (prealloc)
 *           logQ - LxL log matrix of conditional prob matrix (prealloc)
 *
 * Returns:  void.
 */
void
islogQconsistent(FILE *ofp, double *Q_0, double *K, int Lx, int Ly, int Lz, int verbose)
{
  double *R;
  int     i, j;

  R = Cal_M_N_Prod(ofp, Q_0, K, Lx, Ly, Lz, FALSE); /*  multiply Q_0*K, dump it in R */

  for (i = 0; i < Lx; i++)
    for (j = 0; j < Lz; j++) 
      if ( (R[i*Lz+j] < 0.0 && Q_0[i*Lz+j] < 0.0)  )
	Warn ("Ah! Q_0log[Q_0^{-1}*Q] not consistent with a markovian stationary model of evolution (%d,%d: %f %f).", i, j, Q_0[i*Lz+j], R[i*Lz+j]);     
  
  if (verbose) { 
    fprintf(ofp, "RA Instant evolution probabilities\n");
    for (i = 0; i < Lx; i++) {
      for (j = 0; j < Lz; j++) {
	fprintf(ofp, "%.4f ", R[i*Lz+j]);
      }
      fprintf(ofp, "\n");
    }
  }
  
  free(R);
}

/* Function: islogQconsistent_2()
 * Date:     ER, Mon May  8 12:55:29 CDT 2000 [St. Louis]
 *
 * Purpose:  Q(t) ---a matrix of conditional probabilities---
 *           is consistent with being derived from a markovian stationary
 *           model of evolution from:
 *                    Q_0 - a matrix of conditional probabilities at time zero
 *                    A - a matrix of rate of evolution
 *                    K - a matrix of rate of evolution
 *
 *           as
 *                     Q(t) = Q_0 + A[ exp{t K} - Id ]
 *
 *
 *           IF: Q_0 + epsilon * A*K  can play the role of an "instant matrix of evolution"
 *
 *               that is, if all the entries are positive.
 *
 * Args:     Q_0  - LxL matrix of conditional prob matrix at time zero (prealloc)
 *           A    - LxL matrix  (prealloc)
 *           logQ - LxL log matrix of conditional prob matrix (prealloc)
 *
 * Returns:  void.
 */
void
islogQconsistent_2(FILE *ofp, double *Q_0, double *A, double *K, int Lx, int Ly, int Lz, int verbose)
{
  double *R;
  int     i, j;

  R = Cal_M_N_Prod(ofp, A, K, Lx, Ly, Lz, FALSE); /*  multiply A*K, dump it in R */

  for (i = 0; i < Lx; i++)
    for (j = 0; j < Lz; j++)
      if ( (R[i*Lz+j] < 0.0 && Q_0[i*Lz+j] < 0.0)  )
	Warn ("Ah! Q_0log[Q_0^{-1}*Q] not consistent with a markovian stationary model of evolution (%d,%d: %f %f).", i, j, Q_0[i*Lz+j], R[i*Lz+j]);     
  
  if (verbose) { 
    fprintf(ofp, "RA Instant evolution probabilities\n");
    for (i = 0; i < Lx; i++) {
      for (j = 0; j < Lz; j++) {
	fprintf(ofp, "%.4f ", R[i*Lz+j]);
      }
      fprintf(ofp, "\n");
    }
  }
  
  free(R);
}


void
adjust_prob(double *psingle, int size)
{
  int x;
  double sum = 0.0;

  for (x = 0; x < size; x++) {
    if (psingle[x] > 1.0+MARGIN) Die ("adjust_prob(): probabilities are getting too large here. P[%d] = %f", x, psingle[x]);
    if (psingle[x] < -MARGIN2)   Warn ("adjust_prob(): probabilities are getting too small here, P[%d] = %f", x, psingle[x]);

    if (psingle[x] < 0.0 && psingle[x] > - MARGIN2) 
      psingle[x] = 0.0;
    
   sum += psingle[x];
  }

  for (x = 0; x < size; x++) psingle[x] /= sum;
 
}

void
check_reversibility (double *QL, double *QR, double *ml, double *mr, int L)
{
  double *slr;
  double *srl;
  int     i,j;

  slr = (double *) MallocOrDie (sizeof(double) * L * L); 
  srl = (double *) MallocOrDie (sizeof(double) * L * L); 

  for (i = 0; i < L; i++) 
    for (j = 0; j < L; j++) {
      slr[i*L+j] = QL[i*L+j]/mr[j];
      srl[i*L+j] = QR[i*L+j]/ml[j];
    }

    fprintf(stdout, "Slr probabilities\n");
    PrintProbs(stdout, slr, L);
    fprintf(stdout, "Srl probabilities\n");
    PrintProbs(stdout, srl, L);

    for (i = 0; i < L; i++) 
      for (j = 0; j < L; j++) 
	if (slr[i*L+j] - srl[j*L+i] > MARGIN || slr[i*L+j] - srl[j*L+i] < -MARGIN) 
	  Warn("reversibility check failed\n");
    
  free(slr);
  free(srl);
}

void
check_Q_0_reversibility (double *QL, double *QR, double *ml, double *mr, int L, int hasindel)
{
  double *slr;
  double *srl;
  double *pstatl;
  double *pstatr;
  int     dim;
  int     i,j;

  fprintf(stdout, "QL_0 probabilities\n");
  PrintProbs(stdout, QL, L);
  fprintf(stdout, "QR_0 probabilities\n");
  PrintProbs(stdout, QR, L);

  if (hasindel) dim = L-1;
  else          dim = L;
  
  slr    = (double *) MallocOrDie (sizeof(double) * dim * dim); 
  srl    = (double *) MallocOrDie (sizeof(double) * dim * dim); 
  pstatl = (double *) MallocOrDie (sizeof(double) * dim); 
  pstatr = (double *) MallocOrDie (sizeof(double) * dim); 
  
  for (i = 0; i < dim; i++) {
    if (hasindel) pstatl[i] = ml[i]/(1.0-ml[L-1]);
    else          pstatl[i] = ml[i];
    
    if (hasindel) pstatr[i] = mr[i]/(1.0-mr[L-1]);
    else          pstatr[i] = mr[i];
  }
  for (i = 0; i < dim; i++) 
    for (j = 0; j < dim; j++) {
      slr[i*dim+j] = QL[i*L+j]/pstatr[j];
      srl[i*dim+j] = QR[i*L+j]/pstatl[j];
    }
  
  fprintf(stdout, "Slr probabilities\n");
  PrintProbs(stdout, slr, dim);
  fprintf(stdout, "Srl probabilities\n");
  PrintProbs(stdout, srl, dim);
  
  for (i = 0; i < dim; i++) 
    for (j = 0; j < dim; j++) 
      if (slr[i*dim+j] - srl[j*dim+i] > MARGIN || slr[i*dim+j] - srl[j*dim+i] < -MARGIN) 
	Warn("reversibility check failed\n");
  
  free(slr);
  free(srl);
  free(pstatl);
  free(pstatr);
}

int
compare_freqs(double *pml, double *pmr, double *targetfreq, int L)
{
  int flag = FALSE;
  int i;
  
  for (i = 0; i < L; i ++) 
    if (fabs(pml[i]-targetfreq[i]) > 1.0-accuracy1 || fabs(pmr[i]-targetfreq[i]) > 1.0-accuracy1) 
      flag = TRUE;
  
  return flag;
}
