#ifndef GENERALIZED_GRAMMARS_INCLUDED
#define GENERALIZED_GRAMMARS_INCLUDED

#include <stdio.h>

#define MAXNT    8		/* maximum # of nonterminals in the grammar  */
#define MAXRULES 6		/* maximum # of production rules per NT */
#define MAXATOMS 6		/* maximum # of atoms in one production rule */
#define MAXCOORD 2		/* maximum number of residue coords used by any rule */

enum atom_e      { EPSILON_ATOM, 
		   NONTERMINAL_ATOM, 
		   EMISSION_ATOM, 
		   MONOSEGMENT_ATOM, 
		   DISEGMENT_ATOM };

enum iterstate_e { ITERATOR_OFF, 
		   ITERATOR_ON,
		   ITERATOR_TIED };

#define COORD_I  0
#define COORD_J  1
#define COORD_K  2
#define COORD_L  3

#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif
#ifndef MIN
#define MIN(a,b)         (((a)<(b))?(a):(b))
#endif
#ifndef MAX
#define MAX(a,b)         (((a)>(b))?(a):(b))
#endif

typedef int CYKVAL;
#define CYKINFTY    99999999	/* we can hold 20x this in an int without overflow */
#define INTSCALE    1000        /* keep 3 decimal pts precision in log->integer */                    


/* struct rule_s
 * A generalized SCFG production rule.
 * 
 * A rule is an array of "atoms". An atom is an atomic unit of the
 * production; it may be either a single nonterminal, a single
 * terminal, or a correlated group of terminals (e.g. a base pair).
 * There can be up to MAXATOMS atoms in the rule. More specifically,
 * atoms come in five flavors (see atom_e enumerator):
 * EPSILON_ATOM, NONTERMINAL_ATOM, EMISSION_ATOM, MONOSEGMENT_ATOM, and
 * DISEGMENT_ATOM.
 * 
 * An EPSILON_ATOM is a null production, S-> \epsilon. Usually not
 * a good idea to have, because they can easily create structurally
 * ambiguous grammars. nterm_idx, edist_idx, and ldist_idx are all
 * undefined for an EPSILON_ATOM. 
 *
 * A NONTERMINAL_ATOM is a single nonterminal. nterm_idx is the index
 * of which nonterminal in the grammar this is.  edist_idx and
 * ldist_idx are undefined for a NONTERMINAL_ATOM.
 * 
 * An EMISSION_ATOM is one or more terminals. edist_idx is the index
 * of which emission probability distribution to generate these
 * terminals from. nterm_idx and ldist_idx are undefined for an
 * EMISSION_ATOM.
 * 
 * A MONOSEGMENT_ATOM generates a string of terminals in a semi-Markov
 * (duration-modeled) fashion -- appropriate for length modeling of a
 * loop, for example. ldist_idx is the index of the length
 * distribution. edist_idx is the index of a singlet emission
 * distribution that generates individual residues. nterm_idx is
 * undefined.
 * 
 * A DISEGMENT_ATOM generates two nested pairwise-correlated strings
 * of terminals of the same length in a duration-modeled fashion --
 * appropriate for length modeling of a stem. As with monosegments,
 * ldist_idx and edist_idx are indices of the length distribution
 * and the singlet emission distribution, respectively, and nterm_idx
 * is undefined.
 * 
 * Because an atom may be a correlated group of terminals (emitted
 * from a joint distribution), atoms are not necessarily in the same
 * order as symbols on the right hand side of a production rule.
 * That is, for a base pair production S -> a S a', there are two
 * atoms -- an (a,a') EMISSION_ATOM, and an (S) NONTERMINAL_ATOM --
 * and their order is unimportant in the rule_s. 
 * 
 * Instead, the order of the right hand side is encoded in a fashion
 * that facilitates a generalized CYK implementation. Say that the
 * nonterminal on the left hand side of the production will account
 * (in CYK parsing) for a subsequence i..j. Each atom on the right
 * hand side of the production is associated with a coordinate
 * relative to these coordinates i,j. For example, the nonterminal
 * atom in our base pair production S -> a S a' has two coords
 * i+1,j-1, and the emission atom has two coords (for its two emitted
 * bases) i,j. A second example: the singlet emission atom in a
 * production rule S-> a S has one coord i, and the nonterminal atom
 * has coords i+1,j. These are exactly the coords used in dynamic
 * programming recursions in CYK parsing.
 * 
 * Each coord is encoded as a base and an offset. The base (coordbase)
 * is COORD_I, COORD_J, COORD_K, or COORD_L. The offset is an integer
 * offset relative to that base. Thus i+1 is encoded as
 * coordbase=COORD_I, offset=1.
 * 
 * NONTERMINAL_ATOMs always have two coords. EMISSION_ATOMs have one
 * coord per base that they emit. A MONOSEGMENT_ATOM has two coords;
 * a DISEGMENT_ATOM has four coords.
 * 
 * Eventually, atoms will also carry <context>, which is how
 * we'll implement generalized lexicalization; for example, first-order
 * Markov correlations in nearest-neighbor stacking rules. But for
 * now, <context> is an unused placeholder.
 * 
 * COORD_K and COORD_L are generated by d1 and d2 iterators [xref
 * STL8/p130-131].  For a given subsequence i..j of length d; the production rule is 
 * allowed to split it into at most three segments:
 *    i..k //  k+1..l-1  // l+1..j
 *    [d1]      [d3]         [d2]
 * The two iterators support a generalized implementation of
 * bifurcations and other rules that need to take a max over different
 * possible lengths d1 and/or d2. If anyone needs COORD_K, then we
 * need a d1 iterator; if anyone needs COORD_L, then we need a d2
 * iterator; in special case of a DISEGMENT, we need both, but length
 * d2=d1. Monosegment and disegment atoms, which have length
 * distributions, need to set min and max lengths for d1 and/or d2,
 * corresponding to the min and max length in the length distributions.
 * Otherwise, the iterator min's and max's are -1 and MAXINT.
 * 
 * The generalized CYK parser needs to know the minimum subsequence
 * lengths that each atom may generate, for its 
 * initialization/boundary conditions. These are in atom_mind[];
 * they are determined by a dynamic programming algorithm in
 * SetMinDistances(). 
 * 
 * If a rule is composed exclusively of terminals (or epsilon), the
 * <is_allterminals> flag is set to TRUE; in this case, the rule's
 * <mind> (minimum d) is not just the minimum length of this rule,
 * it's the only length.  
 */
typedef struct {
  int  natoms;			  /* number of atoms in the rule            */
  enum atom_e atomtype[MAXATOMS]; /* type (NONTERMINAL_ATOM,  etc.)         */
  int  edist_idx[MAXATOMS];	  /* index of emission dist                 */
  int  ldist_idx[MAXATOMS];	  /* index of length dist (SEGMENTs only)   */
  int  nterm_idx[MAXATOMS];	  /* index of nonterminal (NONTEMINAL only) */

  int  ncoords[MAXATOMS];	      /* number of sequence coords used     */
  int  coordbase[MAXATOMS][MAXCOORD]; /* COORD_I, COORD_J, COORD_K, COORD_L */
  int  offset[MAXATOMS][MAXCOORD];    /* offset of each base coord          */
  int  context[MAXATOMS][MAXCOORD];   /* context to use for each residue    */

  enum iterstate_e d1_iterator_state; /* ITERATOR_ON | _OFF              */
  int  d1_iterator_min;		      /* -1 | minlength of a SEGMENT     */
  int  d1_iterator_max;		      /* MAXINT | maxlength of a SEGMENT */
  enum iterstate_e d2_iterator_state; /* ITERATOR_ON | _OFF | _TIED      */
  int  d2_iterator_min;		      /* -1 | minlength of a SEGMENT     */
  int  d2_iterator_max;		      /* MAXINT | maxlength of a SEGMENT */

  int  atom_mind[MAXATOMS];	/* minimum subseq lengths for each atom  */
  int  is_allterminals;		/* TRUE if rule contains no nonterminals */
} RULE;


/* An emission distribution.
 */
typedef struct {
  int     n;		/* number of residues emitted, usually 1 or 2 */
  double *ep;		/* probability distribution [0..4^n-1]        */
  CYKVAL *esc;		/* scores, [0..4^n-1]                         */
  int     is_basepair;	/* TRUE if n=2 and x,x' are annotated as a bp */
} EDIST;

/* A length distribution.
 */
typedef struct {
  int     min;		/* minimum length w/ prob > 0                  */
  int     max;		/* maximum length w/ prob > 0                  */
  double *lp;		/* probability dist [0..max]; 0..min-1 are 0.0 */
  CYKVAL *lsc;		/* scores, [0..max]; 0..min-1 are -\infty      */
} LDIST;


typedef struct {
  int    M;			/* # of nonterminals in the grammar  */

  RULE   pr[MAXNT][MAXRULES];	/* production rules for each nonterminal */
  int    nrules[MAXNT];
  int    tdist_idx[MAXNT];	/* index of which transition distribution to use for each NT */

  /* Transition distributions carried by this grammar.
   */
  int    ntd;			/* number of transition distributions */
  int    tn[MAXNT];		/* number of rules for each transition distribution */
  double tp[MAXNT][MAXRULES];	/* transition probability distributions */
  CYKVAL tsc[MAXNT][MAXRULES];  /* transition scores (log t[][]) */

  /* Emission distributions carried by this grammar.
   */
  int    ned;
  EDIST *edist;

  /* Length distributions carried by this grammar, if any.
   */
  int    nld;
  LDIST *ldist;

  /* Minimum subsequence lengths allowable for individual rules and
   * nonterminals. Used for initialization/boundary conditions in the
   * generalized parser. Set by a call to SetMinDistances().
   */
  int    nt_mind[MAXNT];	     /* for nonterminals */
  int    rule_mind[MAXNT][MAXRULES]; /* for rules */
} GRAMMAR;


extern GRAMMAR *Grammar_Create(void);
extern void     Grammar_Zero(GRAMMAR *G);
extern void     Grammar_Destroy(GRAMMAR *G);
extern void     Grammar_Dump(FILE *fp, GRAMMAR *G);

extern GRAMMAR *Grammar_CreateNussinovTest(void);

extern void Grammar_SetMinDistances(GRAMMAR *G);

extern void Grammar_Write_Counts(FILE *fp, GRAMMAR *G);
extern void Grammar_Read_Counts(FILE *fp, GRAMMAR *G);
extern void Grammar_Write(FILE *fp, GRAMMAR *G);
extern void Grammar_Read(FILE *fp, GRAMMAR *G);
extern void Grammar_Priorify(GRAMMAR *G);
extern void Grammar_Normalize(GRAMMAR *G);
extern void Grammar_Scorify(GRAMMAR *G);

#endif /* GENERALIZED_GRAMMARS_INCLUDED */
  
