#! /usr/bin/env python3

"""Integration test for `easel ssdraw` 

Usage: easel-ssdraw-itest.py <builddir> <srcdir> <tmppfx>
  <builddir>: path to Easel build dir. `easel` miniapp is <builddir>/miniapps/easel
  <srcdir>:   path to Easel src dir.
  <tmppfx>:   prefix we're allowed to use to create tmp files in current working dir.

These tests aren't very rigorous. They create .ps files, and to test
that a .ps file was created properly, we'd want to look at it in a
postscript viewer. For the purposes of a python test script, we just
check the postscript file for text we know should be there.

Additionally, some of the tests don't use patterns that discriminate
between output with vs. without the option that's being
tested. Improve this whenever possible.
"""

import glob
import os
import re
import sys
import esl_itest

progs_used = [ 'miniapps/easel' ]
files_used = [ ]

(builddir, srcdir, tmppfx) = esl_itest.getargs(sys.argv)
esl_itest.check_files(srcdir,   files_used)
esl_itest.check_progs(builddir, progs_used)

easel = f'{builddir}/miniapps/easel'

##################################################################
# Test files
#
# These were trna-5.stk and trna-ssdraw.ps in old versions of Easel.
# Now pulled into the test script and written as tmpfiles.
#
TRNA5_MSA = """\
# STOCKHOLM 1.0
#=GF AU Infernal 0.1

tRNA1         GCGGAUUUAGCUCAGUuGGG.AGAGCGCCAGACUGAAGAUCUGGAGGuCCUGUGUUCGAUCCACAGAAUUCGCA
#=GR tRNA1 PP ****************9887.*****************************************************
tRNA2         UCCGAUAUAGUGUAAC.GGCuAUCACAUCACGCUUUCACCGUGGAGA.CCGGGGUUCGACUCCCCGUAUCGGAG
#=GR tRNA2 PP ****************.8888**************************.**************************
tRNA3         UCCGUGAUAGUUUAAU.GGUcAGAAUGGGCGCUUGUCGCGUGCCAGA.UCGGGGUUCAAUUCCCCGUCGCGGAG
#=GR tRNA3 PP ****************.******************************.**************************
tRNA4         GCUCGUAUGGCGCAGU.GGU.AGCGCAGCAGAUUGCAAAUCUGUUGGuCCUUAGUUCGAUCCUGAGUGCGAGCU
#=GR tRNA4 PP ****************.***.*****************************************************
tRNA5         ....ACAUGGCGCAGUuGGU.AGCGCGCUUCCCUUGCAAGGAAGAGGuCAUCGGUUCGAUUCCGGUUGC.....
#=GR tRNA5 PP ....****************.************************************************.....
#=GC SS_cons  (((((((,,<<<<___.___._>>>>,<<<<<_______>>>>>,,,.,<<<<<_______>>>>>))))))):
#=GC RF       gccccugUAGcucAaU.GGU.AgagCauuggaCUuuuAAuccaaagg.ugugGGUUCgAaUCCcaccaggggcA
//
"""

SSDRAW_TEMPLATE = """\
% begin ignore
%
% trna-ssdraw.ps:
% An esl-ssdraw postscript template file for drawing
% tRNA secondary structure diagrams. The input alignment
% must be of consensus length (nongap RF length) of 71.
% This corresponds to the Rfam 9.1 RF00005 tRNA model
% (http://rfam.sanger.ac.uk/). That is, this template
% file can be used with alignments generated by Infernal's
% 'cmalign' program using the Rfam 9.1 tRNA CM.
%
% setup defaults
/Helvetica findfont 8.00 scalefont setfont
0.00 0.00 0.00 1.00 setcmykcolor
1.00 setlinewidth
% end ignore

% begin modelname
% tRNA
% end modelname

% begin legend
% 34 -80. -30. 10 0.
% end legend

% begin scale
1.7 1.7 scale
% end scale

% begin regurgitate
/Helvetica findfont 6.00 scalefont setfont
0.00 0.00 0.00 1.00 setcmykcolor
(5') 164.00 400.00 moveto show
(3') 192.00 408.00 moveto show
1.00 setlinewidth
0.00 0.00 0.00 1.00 setcmykcolor
165.50 238.00 188.50 238.00 newpath moveto lineto stroke
166.00 244.00 166.00 238.00 newpath moveto lineto stroke
188.00 244.00 188.00 238.00 newpath moveto lineto stroke
(anticodon) 164.25 230.00 moveto show
% end regurgitate

% begin ignore
% reset 8.0 fontsize
/Helvetica findfont 8.00 scalefont setfont
% end ignore

% begin text positiontext
(20) 124.00 297.00 moveto show
(40) 194.50 287.00 moveto show
(60) 211.00 354.00 moveto show
% end text positiontext

% begin lines positionticks
151.82 331.76 148.86 338.65 newpath moveto lineto stroke
122.05 310.19 124.69 303.17 newpath moveto lineto stroke
168.09 273.73 160.81 275.55 newpath moveto lineto stroke
186.00 289.00 193.50 289.00 newpath moveto lineto stroke
215.00 326.00 215.00 318.50 newpath moveto lineto stroke
215.00 344.00 215.00 351.50 newpath moveto lineto stroke
185.91 394.27 193.19 392.45 newpath moveto lineto stroke
% end lines positionticks

% begin ignore
% set color to grey for bpconnect lines
0.00 0.00 0.00 0.50 setcmykcolor
% end ignore

% begin lines bpconnects
179.00 394.00 175.00 394.00 newpath moveto lineto stroke
179.00 386.00 175.00 386.00 newpath moveto lineto stroke
179.00 378.00 175.00 378.00 newpath moveto lineto stroke
179.00 370.00 175.00 370.00 newpath moveto lineto stroke
179.00 362.00 175.00 362.00 newpath moveto lineto stroke
179.00 354.00 175.00 354.00 newpath moveto lineto stroke
179.00 346.00 175.00 346.00 newpath moveto lineto stroke
153.00 321.00 153.00 325.00 newpath moveto lineto stroke
145.00 321.00 145.00 325.00 newpath moveto lineto stroke
137.00 321.00 137.00 325.00 newpath moveto lineto stroke
129.00 321.00 129.00 325.00 newpath moveto lineto stroke
179.00 304.00 175.00 304.00 newpath moveto lineto stroke
179.00 296.00 175.00 296.00 newpath moveto lineto stroke
179.00 288.00 175.00 288.00 newpath moveto lineto stroke
179.00 280.00 175.00 280.00 newpath moveto lineto stroke
179.00 272.00 175.00 272.00 newpath moveto lineto stroke
191.00 337.00 191.00 333.00 newpath moveto lineto stroke
199.00 337.00 199.00 333.00 newpath moveto lineto stroke
207.00 337.00 207.00 333.00 newpath moveto lineto stroke
215.00 337.00 215.00 333.00 newpath moveto lineto stroke
223.00 337.00 223.00 333.00 newpath moveto lineto stroke
% end lines bpconnects

% begin ignore
% reset color to black
0.00 0.00 0.00 1.00 setcmykcolor
% end ignore

% begin text nucleotides
(G) 168.00 392.00 moveto show
(C) 168.00 384.00 moveto show
(G) 168.00 376.00 moveto show
(G) 168.00 368.00 moveto show
(A) 168.00 360.00 moveto show
(U) 168.00 352.00 moveto show
(U) 168.00 344.00 moveto show
(U) 162.00 338.00 moveto show
(A) 156.00 332.00 moveto show
% nucleotide 10 on next line
(G) 150.00 326.00 moveto show
(C) 142.00 326.00 moveto show
(U) 134.00 326.00 moveto show
(C) 126.00 326.00 moveto show
(A) 118.00 330.00 moveto show
(G) 110.00 332.00 moveto show
(U) 102.00 328.00 moveto show
(U) 99.00 320.00 moveto show
(G) 102.00 312.00 moveto show
(G) 110.00 308.00 moveto show
% nucleotide 20 on next line
(A) 118.00 310.00 moveto show
(G) 126.00 314.00 moveto show
(A) 134.00 314.00 moveto show
(G) 142.00 314.00 moveto show
(C) 150.00 314.00 moveto show
(G) 159.00 308.00 moveto show
(C) 168.00 302.00 moveto show
(C) 168.00 294.00 moveto show
(A) 168.00 286.00 moveto show
(G) 168.00 278.00 moveto show
% nucleotide 30 on next line
(A) 168.00 270.00 moveto show
(C) 164.00 262.00 moveto show
(U) 162.00 254.00 moveto show
(G) 166.00 246.00 moveto show
(A) 174.00 243.00 moveto show
(A) 182.00 246.00 moveto show
(G) 186.00 254.00 moveto show
(A) 184.00 262.00 moveto show
(U) 180.00 270.00 moveto show
(C) 180.00 278.00 moveto show
% nucleotide 40 on next line
(U) 180.00 286.00 moveto show
(G) 180.00 294.00 moveto show
(G) 180.00 302.00 moveto show
(A) 188.00 306.00 moveto show
(G) 196.00 306.00 moveto show
(U) 200.00 314.00 moveto show
(U) 192.00 318.00 moveto show
(C) 188.00 326.00 moveto show
(U) 196.00 326.00 moveto show
(G) 204.00 326.00 moveto show
% nucleotide 50 on next line
(U) 212.00 326.00 moveto show
(G) 220.00 326.00 moveto show
(U) 228.00 322.00 moveto show
(U) 236.00 320.00 moveto show
(C) 244.00 324.00 moveto show
(G) 247.00 332.00 moveto show
(A) 244.00 340.00 moveto show
(U) 236.00 344.00 moveto show
(C) 228.00 342.00 moveto show
(C) 220.00 338.00 moveto show
% nucleotide 60 on next line
(A) 212.00 338.00 moveto show
(C) 204.00 338.00 moveto show
(A) 196.00 338.00 moveto show
(G) 188.00 338.00 moveto show
(A) 180.00 344.00 moveto show
(A) 180.00 352.00 moveto show
(U) 180.00 360.00 moveto show
(U) 180.00 368.00 moveto show
(C) 180.00 376.00 moveto show
(G) 180.00 384.00 moveto show
% nucleotide 70 on next line
(C) 180.00 392.00 moveto show
(A) 184.00 400.00 moveto show
% end text nucleotides

% begin ignore
showpage
% end ignore
"""

MASK1 = """\
10010110101101111111111010101111111010101011010001100010101110101010101
"""

MASK2 = """\
11011011111110101011111111111010101010100000000001001010011101010101010
"""

IFILE = """\
tRNA 71
tRNA1 73 1 71  16 17 1  45 47 1
tRNA2 72 1 71  19 20 1
tRNA3 72 1 71  19 20 1
tRNA4 72 1 71  45 46 1
tRNA5 64 5 71  16 13 1  45 43 1
//
"""

DFILE = """\
Example of using --dfile to specify colors
These numbers mean what?
0.0 0.6 0.7 0.8 0.9 0.95 1.0
0.560
0.730
0.210
0.120
0.910
1.000
0.000
0.302
0.121
1.000
0.998
0.353
0.5
0.342
0.560
0.730
0.210
0.120
0.910
1.000
0.000
0.302
0.121
1.000
0.998
0.353
0.5
0.342
0.560
0.730
0.210
0.120
0.910
1.000
0.000
0.302
0.121
1.000
0.998
0.353
0.5
0.342
0.560
0.730
0.210
0.120
0.910
1.000
0.000
0.302
0.121
1.000
0.998
0.353
0.5
0.342
0.560
0.730
0.210
0.120
0.910
1.000
0.000
0.302
0.121
1.000
0.998
0.353
0.5
0.342
0.342
//
Another example of using --dfile to specify colors
Meaning is unknown
0.0 0.1 0.2 0.3 0.4 0.5 1.0
0.560
0.730
0.210
0.120
0.910
1.000
0.000
0.302
0.121
1.000
0.998
0.353
0.5
0.342
0.560
0.730
0.210
0.120
0.910
1.000
0.000
0.302
0.121
1.000
0.998
0.353
0.5
0.342
0.560
0.730
0.210
0.120
0.910
1.000
0.000
0.302
0.121
1.000
0.998
0.353
0.5
0.342
0.560
0.730
0.210
0.120
0.910
1.000
0.000
0.302
0.121
1.000
0.998
0.353
0.5
0.342
0.560
0.730
0.210
0.120
0.910
1.000
0.000
0.302
0.121
1.000
0.998
0.353
0.5
0.342
0.342
//
"""

EFILE = """\
0.228 0.636 0.044 0.680
0.470 0.564 0.932 0.933
0.034 0.171 0.204 0.137
0.471 0.725 0.013 0.246
0.414 0.111 0.478 0.329
0.947 0.110 0.870 0.230
0.789 0.964 0.649 0.174
0.991 0.362 0.028 0.634
0.977 0.838 0.265 0.046
0.024 0.806 0.148 0.647
0.235 0.891 0.594 0.694
0.799 0.366 0.073 0.013
0.914 0.393 0.793 0.826
0.054 0.786 0.486 0.386
0.794 0.360 0.061 0.304
0.070 0.866 0.646 0.961
0.217 0.462 0.220 0.608
0.446 0.715 0.516 0.287
0.546 0.924 0.456 0.468
0.438 0.035 0.613 0.031
0.321 0.437 0.935 0.336
0.101 0.275 0.089 0.410
0.714 0.681 0.622 0.676
0.315 0.079 0.850 0.917
0.524 0.897 0.908 0.902
0.460 0.826 0.039 0.979
0.041 0.130 0.599 0.122
0.181 0.234 0.813 0.157
0.514 0.145 0.386 0.801
0.183 0.929 0.161 0.588
0.852 0.746 0.189 0.836
0.455 0.580 0.439 0.373
0.473 0.212 0.655 0.536
0.678 0.744 0.434 0.793
0.844 0.816 0.646 0.459
0.806 0.807 0.002 0.304
0.395 0.972 0.800 0.119
0.981 0.492 0.752 0.463
0.926 0.327 0.066 0.099
0.029 0.252 0.846 0.134
0.571 0.306 0.698 0.213
0.711 0.470 0.680 0.522
0.639 0.930 0.336 0.968
0.839 0.824 0.118 0.877
0.399 0.449 0.675 0.740
0.105 0.341 0.014 0.971
0.965 0.277 0.691 0.732
0.198 0.648 0.903 0.804
0.958 0.939 0.561 0.028
0.359 0.665 0.162 0.298
0.581 0.260 0.237 0.773
0.042 0.540 0.410 0.190
0.013 0.841 0.068 0.796
0.763 0.904 0.601 0.677
0.849 0.928 0.010 0.797
0.517 0.658 0.094 0.870
0.378 0.132 0.736 0.559
0.385 0.358 0.951 0.566
0.100 0.141 0.397 0.492
0.091 0.913 0.870 0.351
0.819 0.429 0.862 0.267
0.263 0.777 0.427 0.602
0.005 0.225 0.023 0.157
0.154 0.009 0.800 0.514
0.915 0.456 0.162 0.235
0.374 0.595 0.678 0.901
0.099 0.163 0.410 0.515
0.969 0.191 0.912 0.117
0.557 0.919 0.084 0.690
0.846 0.032 0.300 0.158
0.736 0.131 0.007 0.138
//
0.0 0.0 0.0 0.0 T
0.0 0.0 0.0 0.0 h
0.0 0.0 0.0 0.0 i
0.0 0.0 0.0 0.0 s
0.0 0.0 0.0 0.0 i
0.0 0.0 0.0 0.0 s
0.0 0.0 0.0 0.0 a
0.0 0.0 0.0 0.0 t
0.0 0.0 0.0 0.0 R
0.0 0.0 0.0 0.0 N
0.0 0.0 0.0 0.0 A
0.0 0.0 0.0 0.0 T
0.0 0.0 0.0 0.0 h
0.0 0.0 0.0 0.0 i
0.0 0.0 0.0 0.0 s
0.0 0.0 0.0 0.0 i
0.0 0.0 0.0 0.0 s
0.0 0.0 0.0 0.0 a
0.0 0.0 0.0 0.0 t
0.0 0.0 0.0 0.0 R
0.0 0.0 0.0 0.0 N
0.0 0.0 0.0 0.0 A
0.0 0.0 0.0 0.0 T
0.0 0.0 0.0 0.0 h
0.0 0.0 0.0 0.0 i
0.0 0.0 0.0 0.0 s
0.0 0.0 0.0 0.0 i
0.0 0.0 0.0 0.0 s
0.0 0.0 0.0 0.0 a
0.0 0.0 0.0 0.0 t
0.0 0.0 0.0 0.0 R
0.0 0.0 0.0 0.0 N
0.0 0.0 0.0 0.0 A
0.0 0.0 0.0 0.0 T
0.0 0.0 0.0 0.0 h
0.0 0.0 0.0 0.0 i
0.0 0.0 0.0 0.0 s
0.0 0.0 0.0 0.0 i
0.0 0.0 0.0 0.0 s
0.0 0.0 0.0 0.0 a
0.0 0.0 0.0 0.0 t
0.0 0.0 0.0 0.0 R
0.0 0.0 0.0 0.0 N
0.0 0.0 0.0 0.0 A
0.0 0.0 0.0 0.0 T
0.0 0.0 0.0 0.0 h
0.0 0.0 0.0 0.0 i
0.0 0.0 0.0 0.0 s
0.0 0.0 0.0 0.0 i
0.0 0.0 0.0 0.0 s
0.0 0.0 0.0 0.0 a
0.0 0.0 0.0 0.0 t
0.0 0.0 0.0 0.0 R
0.0 0.0 0.0 0.0 N
0.0 0.0 0.0 0.0 A
0.0 0.0 0.0 0.0 T
0.0 0.0 0.0 0.0 h
0.0 0.0 0.0 0.0 i
0.0 0.0 0.0 0.0 s
0.0 0.0 0.0 0.0 i
0.0 0.0 0.0 0.0 s
0.0 0.0 0.0 0.0 a
0.0 0.0 0.0 0.0 t
0.0 0.0 0.0 0.0 R
0.0 0.0 0.0 0.0 N
0.0 0.0 0.0 0.0 A
0.0 0.0 0.0 0.0 T
0.0 0.0 0.0 0.0 h
0.0 0.0 0.0 0.0 i
0.0 0.0 0.0 0.0 s
0.0 0.0 0.0 0.0 i
//
"""

with open(f'{tmppfx}.sto',   'w', encoding='utf-8') as f:
    f.write(TRNA5_MSA)
with open(f'{tmppfx}.t',     'w', encoding='utf-8') as f:
    f.write(SSDRAW_TEMPLATE)
with open(f'{tmppfx}.mask1', 'w', encoding='utf-8') as f:
    f.write(MASK1)
with open(f'{tmppfx}.mask2', 'w', encoding='utf-8') as f:
    f.write(MASK2)
with open(f'{tmppfx}.ifile', 'w', encoding='utf-8') as f:
    f.write(IFILE)
with open(f'{tmppfx}.dfile', 'w', encoding='utf-8') as f:
    f.write(DFILE)
with open(f'{tmppfx}.efile', 'w', encoding='utf-8') as f:
    f.write(EFILE)

# end of test tmpfiles. resume code...
################################################################

# `-h` help
r = esl_itest.run(f'{easel} ssdraw -h')

# For most tests, we run the same test with and without --small
#
# optcheck assertion: --small
for smallopt in ['', '--small']:
    # basic
    s = esl_itest.run(f'{easel} ssdraw {smallopt} {tmppfx}.sto {tmppfx}.t {tmppfx}.ps',
                      outfile=f'{tmppfx}.ps')
    if not re.search(r'^\(tRNA\s+71\s+21\)', s, flags=re.MULTILINE):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')
    if not re.search(r'^\(\\\[0\.800-1\.200\\\)\)', s, flags=re.MULTILINE):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')
    if not re.search(r'^%nucleotide 9\nnewpath\n.+\n\s+0\.00\d* 0\.21\d* 1\.00\d* 0\.00\d* setcmyk',
                     s, flags=re.MULTILINE):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')

    # --cons      draw diagram showing the alignment's consensus sequence
    #             (We just test that this runs w/out crashing. EPN didn't test it at all.)
    s = esl_itest.run(f'{easel} ssdraw --cons {smallopt} {tmppfx}.sto {tmppfx}.t {tmppfx}.ps',
                      outfile=f'{tmppfx}.ps')

    # -d          draw default set of alignment summary diagrams
    s = esl_itest.run(f'{easel} ssdraw -d {smallopt} {tmppfx}.sto {tmppfx}.t {tmppfx}.ps',
                      outfile=f'{tmppfx}.ps')
    if not re.search(r'^\(tRNA\s+71\s+21\)', s, flags=re.MULTILINE):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')
    if not re.search(r'^\(\\\[0\.800-1\.200\\\)\)', s, flags=re.MULTILINE):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')
    if not re.search(r'^%nucleotide 9\nnewpath\n.+\n\s+0\.00\d* 0\.21\d* 1\.00\d* 0\.00\d* setcmyk',
                     s, flags=re.MULTILINE):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')

    # --prob      draw average posterior probability diagram
    s = esl_itest.run(f'{easel} ssdraw --prob {smallopt} {tmppfx}.sto {tmppfx}.t {tmppfx}.ps',
                      outfile=f'{tmppfx}.ps')
    if not re.search(r'^\(tRNA\s+71\s+21\)', s, flags=re.MULTILINE):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')
    if not re.search(r'^\(\\\[0\.900-0\.950\\\)\)', s, flags=re.MULTILINE):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')
    if not re.search(r'%nucleotide 18\nnewpath\n.+\n\s+0\.50\d* 0\.00\d* 0\.00\d* 0\.50\d* setcmyk',
                     s, flags=re.MULTILINE):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')

    # --ifreq     draw insert frequency diagram
    s = esl_itest.run(f'{easel} ssdraw --ifreq {smallopt} {tmppfx}.sto {tmppfx}.t {tmppfx}.ps',
                      outfile=f'{tmppfx}.ps')
    if not re.search(r'^\(tRNA\s+71\s+21\)', s, flags=re.MULTILINE):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')
    if not re.search(r'^\(\\\[0\.500-1\.000\\\]\)', s, flags=re.MULTILINE):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')
    if not re.search(r'^%nucleotide 19\nnewpath\n'
                     r'.+0 8 rlineto 8 0 rlineto 0 -8 rlineto closepath\n'
                     r'\s+0\.00\d* 0\.63\d* 1\.00\d* 0\.00\d* setcmykcolor',
                     s, flags=re.MULTILINE):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')

    # --iavglen   draw average insert length diagram
    s = esl_itest.run(f'{easel} ssdraw --iavglen {smallopt} {tmppfx}.sto {tmppfx}.t {tmppfx}.ps',
                      outfile=f'{tmppfx}.ps')
    if not re.search(r'^\(tRNA\s+71\s+21\)', s, flags=re.MULTILINE):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')
    if not re.search(r'^\(>= 10\)', s, flags=re.MULTILINE):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')
    if not re.search(r'%nucleotide 19\nnewpath\n'
                     r'.+moveto  0 8 rlineto 8 0 rlineto 0 -8 rlineto closepath\n'
                     r'\s+0\.92\d* 0\.84\d* 0\.00\d* 0\.08\d* setcmykcolor',
                     s, flags=re.MULTILINE):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')

    # --dall      draw delete diagram w/all deletions (incl. terminal deletes)
    s = esl_itest.run(f'{easel} ssdraw --dall {smallopt} {tmppfx}.sto {tmppfx}.t {tmppfx}.ps',
                      outfile=f'{tmppfx}.ps')
    if not re.search(r'^\(tRNA\s+71\s+21\)', s, flags=re.MULTILINE):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')
    if not re.search(r'\(zero deletions\)', s, flags=re.MULTILINE):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')
    if not re.search(r'%nucleotide 1\nnewpath\n'
                     r'.+moveto  0 8 rlineto 8 0 rlineto 0 -8 rlineto closepath\n'
                     r'\s+0\.42\d* 0\.00\d* 1\.00\d* 0\.00\d* setcmykcolor',
                     s, flags=re.MULTILINE):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')

    # --dint      draw delete diagram w/only internal (non-terminal) deletions
    s = esl_itest.run(f'{easel} ssdraw --dint {smallopt} {tmppfx}.sto {tmppfx}.t {tmppfx}.ps',
                      outfile=f'{tmppfx}.ps')
    if not re.search(r'^\(tRNA\s+71\s+21\)', s, flags=re.MULTILINE):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')
    if not re.search(r'\(zero internal deletions\)', s, flags=re.MULTILINE):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')
    if not re.search(r'%nucleotide 1\nnewpath\n'
                     r'.+moveto  0 8 rlineto 8 0 rlineto 0 -8 rlineto closepath\n'
                     r'\s+0\.00\d* 0\.00\d* 0\.00\d* 0\.20\d* setcmykcolor',
                     s, flags=re.MULTILINE):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')

    # --mutinfo   draw base pair mutual information diagram
    s = esl_itest.run(f'{easel} ssdraw --mutinfo {smallopt} {tmppfx}.sto {tmppfx}.t {tmppfx}.ps',
                      outfile=f'{tmppfx}.ps')
    if not re.search(r'^\(tRNA\s+71\s+21\)', s, flags=re.MULTILINE):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')
    if not re.search(r'\(0 complete basepairs\)', s, flags=re.MULTILINE):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')
    if not re.search(r'%nucleotide 61\nnewpath\n'
                     r'.+moveto  0 8 rlineto 8 0 rlineto 0 -8 rlineto closepath\n'
                     r'\s+0\.42\d* 0\.00\d* 1\.00\d* 0\.00\d* setcmykcolor',
                     s, flags=re.MULTILINE):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')

    # --span      draw diagram showing fraction of seqs that span each posn
    s = esl_itest.run(f'{easel} ssdraw --span {smallopt} {tmppfx}.sto {tmppfx}.t {tmppfx}.ps',
                      outfile=f'{tmppfx}.ps')
    if not re.search(r'^\(tRNA\s+71\s+21\)', s, flags=re.MULTILINE):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')
    if not re.search(r'\(no sequences span\)', s, flags=re.MULTILINE):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')
    if not re.search(r'%nucleotide 2\nnewpath\n'
                     r'.+0 8 rlineto 8 0 rlineto 0 -8 rlineto closepath\n'
                     r'\s+0\.50\d* 0\.00\d* 0\.00\d* 0\.50\d* setcmykcolor',
                     s, flags=re.MULTILINE):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')

    # --info      draw information content diagram
    # --prob      draw average posterior probability diagram
    # --tabfile   s per position data in tabular format to file <f>
    #
    # optcheck assertion: --tabfile
    # (This is just to get line length down for pylint)
    s = esl_itest.run(f'{easel} ssdraw --info --prob --ifreq --dall --dint --mutinfo --span '
                      f'--tabfile {tmppfx}.tab '
                      f'{smallopt} {tmppfx}.sto {tmppfx}.t {tmppfx}.ps',
                      outfile=f'{tmppfx}.ps')
    if not re.search(r'setcmykcolor \(g\).+moveto show', s):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')
    with open(f'{tmppfx}.tab', encoding='utf-8') as f:
        s = f.read()
    for testpattern in [ r'Information content data',  r'Mutual information data',
                         r'Insert frequency data',     r'Delete data',
                         r'Internal delete data',      r'Average posterior probability data',
                         r'Span data',                 r'infocontent\s+7\s+1\.27\d+\s+5\s+4',
                         r'mutualinfo\s+16\s+30\s+38\s+0\.47\d*\s+0\.47\d*\s+0\.76\d*\s+5\s+5',
                         r'insertfreq\s+45\s+0\.60000\s+5\s+6',
                         r'deleteall\s+67\s+0\.20\d*\s+4',
                         r'deleteint\s+67\s+0\.00\d*\s+4\s+0',
                         r'avgpostprob\s+19\s+0\.88\d*\s+5\s+4',
                         r'span\s+67\s+0.80\d*\s+6' ]:
        if not re.search(testpattern, s):
            esl_itest.fail(msg=f'with pattern {testpattern}\nwith extra opts: {smallopt}')

    # --rf        draw diagram showing reference (#=GC RF) sequence
    s = esl_itest.run(f'{easel} ssdraw --rf {smallopt} {tmppfx}.sto {tmppfx}.t {tmppfx}.ps',
                      outfile=f'{tmppfx}.ps')
    if not re.search(r'\(\*REFERENCE\* \(\"#=GC RF\"\)\) 212\.49 418\.24 moveto show', s):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')
    if not re.search(r'\(a\) 110\.00 332\.00 moveto show', s):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')

    # --mask
    # --mask-col    w/--mask draw two color diagram denoting masked columns
    s = esl_itest.run(f'{easel} ssdraw --mask-col --mask {tmppfx}.mask1 '
                      f'{smallopt} {tmppfx}.sto {tmppfx}.t {tmppfx}.ps',
                      outfile=f'{tmppfx}.ps')
    if not re.search(r'%nucleotide 3\n'
                     r'newpath\n'
                     r'\s+166.40 374.40 moveto\s+0 8 rlineto 8 0 rlineto 0 -8 rlineto closepath\n'
                     r'\s+0.00\d* 1.00\d* 0.00\d* 0.00\d* setcmykcolor', s, flags=re.MULTILINE):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')

    # --mask-diff   with --mask <f1>, compare mask in <f1> to mask in <f>
    s = esl_itest.run(f'{easel} ssdraw --mask-col --mask {tmppfx}.mask1 --mask-diff {tmppfx}.mask2 '
                      f'{smallopt} {tmppfx}.sto {tmppfx}.t {tmppfx}.ps',
                      outfile=f'{tmppfx}.ps')
    if not re.search(r'%nucleotide 3\nnewpath\n'
                     r'\s+166.40 374.40 moveto\s+0 8 rlineto 8 0 rlineto 0 -8 rlineto closepath\n'
                     r'\s+0.00\d* 0.00\d* 0.00\d* 0.20\d* setcmykcolor', s):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')

    # --no-leg    do not draw legend
    s = esl_itest.run(f'{easel} ssdraw --no-leg {smallopt} {tmppfx}.sto {tmppfx}.t {tmppfx}.ps',
                      outfile=f'{tmppfx}.ps')
    if re.search(r'\(LEGEND\) 134\.00 216\.00 moveto show', s):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')

    # --no-head   do not draw header
    s = esl_itest.run(f'{easel} ssdraw --no-head {smallopt} {tmppfx}.sto {tmppfx}.t {tmppfx}.ps',
                      outfile=f'{tmppfx}.ps')
    if re.search(r'\(tRNA\s+71\s+21\) 96\.98 418\.24 moveto show', s):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')

    # --no-foot   do not draw footer
    s = esl_itest.run(f'{easel} ssdraw --no-foot {smallopt} {tmppfx}.sto {tmppfx}.t {tmppfx}.ps',
                      outfile=f'{tmppfx}.ps')
    if re.search(r"\(Created by 'esl-ssdraw'. Copyright \(C\) 2\d\d\d", s):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')

    # --no-cnt      do not draw consensus nts on alignment summary diagrams
    s = esl_itest.run(f'{easel} ssdraw --info --no-cnt '
                      f'{smallopt} {tmppfx}.sto {tmppfx}.t {tmppfx}.ps',
                      outfile=f'{tmppfx}.ps')
    if re.search(r'^\(Consensus nucleotides \(nt\) are displayed', s, flags=re.MULTILINE):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')

    # --cthresh     capitalize cons nts occurring in >= <x> fraction of seqs
    s = esl_itest.run(f'{easel} ssdraw --info --cthresh 0.4 '
                      f'{smallopt} {tmppfx}.sto {tmppfx}.t {tmppfx}.ps',
                      outfile=f'{tmppfx}.ps')
    if not re.search(r'^\(Capitalized nts occur in >= 0\.40 fraction of sequences\)',
                     s, flags=re.MULTILINE):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')
    if not re.search(r'setcmykcolor \(G\) 168\.\d+ 392\.\d+ moveto show', s):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')

    # --cambig      allow ambiguous nts in consensus sequence
    s = esl_itest.run(f'{easel} ssdraw --info --cambig '
                      f'{smallopt} {tmppfx}.sto {tmppfx}.t {tmppfx}.ps',
                      outfile=f'{tmppfx}.ps')
    if not re.search(r'^\(as the least ambiguous nt that represents >= 0.9',
                     s, flags=re.MULTILINE):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')
    if not re.search(r'setcmykcolor \(K\) 168\.\d+ 392\.\d+ moveto show', s):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')

    # --athresh     w/--cambig, cons nt must represent >= <x> fraction of seqs
    s = esl_itest.run(f'{easel} ssdraw --info --cambig --athresh 0.5 '
                      f'{smallopt} {tmppfx}.sto {tmppfx}.t {tmppfx}.ps',
                      outfile=f'{tmppfx}.ps')
    if not re.search(r'^\(as the least ambiguous nt that represents >= 0.5',
                     s, flags=re.MULTILINE):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')
    if not re.search(r'setcmykcolor \(M\) 196.00 338.00 moveto show', s):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')

    # --mask-x      with --mask, mark masked columns as x's
    # --mask-u      with --mask, mark masked columns as squares
    # --mask-a      with --mask-u or --mask-x, draw alternative mask style
    s = esl_itest.run(f'{easel} ssdraw --mask {tmppfx}.mask1 '
                      f'{smallopt} {tmppfx}.sto {tmppfx}.t {tmppfx}.ps',
                      outfile=f'{tmppfx}.ps')
    if not re.search(r'%nucleotide 3\nnewpath\n 171.00 379.00 3.0 0 360 arc closepath\n'
                     r'\s+0.0\d*0 0.63\d* 1.0\d*0 0.0\d*0 setcmykcolor\n\s+stroke', s):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')

    s = esl_itest.run(f'{easel} ssdraw --mask-x --mask {tmppfx}.mask1 '
                      f'{smallopt} {tmppfx}.sto {tmppfx}.t {tmppfx}.ps',
                      outfile=f'{tmppfx}.ps')
    if not re.search(r'stroke\n  175.00 375.00 moveto  -8.0 8.0 rlineto closepath', s):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')

    s = esl_itest.run(f'{easel} ssdraw --mask-u --mask {tmppfx}.mask1 '
                      f'{smallopt} {tmppfx}.sto {tmppfx}.t {tmppfx}.ps',
                      outfile=f'{tmppfx}.ps')
    if not re.search(r'^%nucleotide 3\nnewpath\n'
                     r'\s+168.00 376.00 moveto  0 6.0 rlineto '
                     r'6.0 0 rlineto 0 -6.0 rlineto closepath\n'
                     r'\s+0.00\d* 0.63\d* 1.00\d* 0.00\d* setcmykcolor',
                     s, flags=re.MULTILINE):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')

    s = esl_itest.run(f'{easel} ssdraw --mask-a --mask-x --mask {tmppfx}.mask1 '
                      f'{smallopt} {tmppfx}.sto {tmppfx}.t {tmppfx}.ps',
                      outfile=f'{tmppfx}.ps')
    if not re.search(r'^%nucleotide 3\nnewpath\n'
                     r'\s+0.00\d* 0.63\d* 1.00\d* 0.00\d* setcmykcolor\n'
                     r'\s+167.00 375.00 moveto  8.0 8.0 rlineto closepath\n  stroke',
                     s, flags=re.MULTILINE):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')

    s = esl_itest.run(f'{easel} ssdraw --mask-a --mask-u --mask {tmppfx}.mask1 '
                      f'{smallopt} {tmppfx}.sto {tmppfx}.t {tmppfx}.ps',
                      outfile=f'{tmppfx}.ps')
    if not re.search(r'^%nucleotide 3\nnewpath\n'
                     r'\s+168.50 376.50 moveto  0 5.0 rlineto '
                     r'5.0 0 rlineto 0 -5.0 rlineto closepath\n'
                     r'\s+0.00\d* 0.63\d* 1.00\d* 0.00\d* setcmykcolor',
                     s, flags=re.MULTILINE):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')

    # --dfile       read 'draw file' specifying >=1 diagrams
    s = esl_itest.run(f'{easel} ssdraw --dfile {tmppfx}.dfile '
                      f'{smallopt} {tmppfx}.sto {tmppfx}.t {tmppfx}.ps',
                      outfile=f'{tmppfx}.ps')
    if not re.search(r'^\(Example of using --dfile to\)', s, flags=re.MULTILINE):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')

    s = esl_itest.run(f'{easel} ssdraw --mask {tmppfx}.mask1 --dfile {tmppfx}.dfile '
                      f'{smallopt} {tmppfx}.sto {tmppfx}.t {tmppfx}.ps',
                      outfile=f'{tmppfx}.ps')
    if not re.search(r'^%nucleotide 3\nnewpath\n 171.00 379.00 3.0 0 360 arc closepath\n'
                     r'\s+0.9200 0.8400 0.00\d* 0.0800 setcmykcolor',
                     s, flags=re.MULTILINE):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')

    # --efile       read 'expert draw file' specifying >=1 diagrams
    s = esl_itest.run(f'{easel} ssdraw --efile {tmppfx}.efile '
                      f'{smallopt} {tmppfx}.sto {tmppfx}.t {tmppfx}.ps',
                      outfile=f'{tmppfx}.ps')
    if not re.search(r'^%nucleotide 3\nnewpath\n'
                     r'\s+166.40 374.40 moveto  0 8 rlineto 8 0 rlineto 0 -8 rlineto closepath\n'
                     r'\s+0.03 0.17 0.20 0.14 setcmykcolor',
                     s, flags=re.MULTILINE):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')

    # --ifile       read insert information from cmalign insert file <f>
    s = esl_itest.run(f'{easel} ssdraw --ifreq --ifile {tmppfx}.ifile '
                      f'{smallopt} {tmppfx}.sto {tmppfx}.t {tmppfx}.ps',
                      outfile=f'{tmppfx}.ps')
    if not re.search(r'^%nucleotide 45\nnewpath\n'
                     r'\s+198.40 312.40 moveto  0 8 rlineto 8 0 rlineto 0 -8 rlineto closepath\n'
                     r'\s+0.00\d* 0.94\d* 1.00\d* 0.00\d* setcmykcolor',
                     s, flags=re.MULTILINE):
        esl_itest.fail(msg=f'with extra opts: {smallopt}')

################################################################
# Now for remaining tests that are incompatible with --small
#
# --indi      draw diagrams for individual sequences in the alignment
# -f          force; w/--indi draw all seqs, even if predicted s >100 Mb
s = esl_itest.run(f'{easel} ssdraw --indi -f {tmppfx}.sto {tmppfx}.t {tmppfx}.ps',
                  outfile=f'{tmppfx}.ps')
for testpattern in [ r'tRNA1', r'tRNA2', r'tRNA3', r'tRNA4', r'tRNA5',
                     r'\(A\) 180.00 392.00 moveto show']:
    if not re.search(testpattern, s):
        esl_itest.fail(msg=f'with pattern {testpattern}')
if not re.search(r'%nucleotide 19\nnewpath\n'
                 r'\s+108.40 306.40 moveto\s+0 8 rlineto 8 0 rlineto 0 -8 rlineto closepath\n'
                 r'\s+0.50\d* 0.00\d* 0.00\d* 0.50\d* setcmykcolor', s):
    esl_itest.fail()
if not re.search(r'%nucleotide 19\nnewpath\n'
                 r'\s+108.40 306.40 moveto\s+0 8 rlineto 8 0 rlineto 0 -8 rlineto closepath\n'
                 r'\s+0.00\d* 0.21\d* 1.00\d* 0.00\d* setcmykcolor', s):
    esl_itest.fail()

# --no-pp       with --indi, do not draw indi posterior probability diagrams
s = esl_itest.run(f'{easel} ssdraw --no-pp --indi {tmppfx}.sto {tmppfx}.t {tmppfx}.ps',
                  outfile=f'{tmppfx}.ps')
for testpattern in [ r'tRNA1', r'tRNA2', r'tRNA3', r'tRNA4', r'tRNA5',
                     r'\(A\) 180.00 392.00 moveto show',
                     r'\(Watson-Crick',
                     r'\(Positions !=' ]:
    if not re.search(testpattern, s):
        esl_itest.fail(msg=f'with pattern {testpattern}')
if not re.search(r'%nucleotide 19\nnewpath\n'
                 r'\s+108.40 306.40 moveto\s+0 8 rlineto 8 0 rlineto 0 -8 rlineto closepath\n'
                 r'\s+0.50\d* 0.00\d* 0.00\d* 0.50\d* setcmykcolor', s):
    esl_itest.fail()
if re.search(r'^%nucleotide 19\nnewpath\n'
             r'\s+108.40 306.40 moveto  0 8 rlineto 8 0 rlineto 0 -8 rlineto closepath\n'
             r'\s+0.92\d* 0.84\d* 0.00\d* 0.08\d* setcmykcolor',
             s, flags=re.MULTILINE):
    esl_itest.fail()

# --no-bp       do not color nucleotides based on basepair type
s = esl_itest.run(f'{easel} ssdraw --no-bp --indi {tmppfx}.sto {tmppfx}.t {tmppfx}.ps',
                  outfile=f'{tmppfx}.ps')
for testpattern in [ r'tRNA1', r'tRNA2', r'tRNA3', r'tRNA4', r'tRNA5',
                     r'\(A\) 180.00 392.00 moveto show',
                     r'\(Positions !=' ]:
    if not re.search(testpattern, s):
        esl_itest.fail(msg=f'with pattern {testpattern}')
if re.search(r'\(Watson-Crick', s):
    esl_itest.fail()
if not re.search(r'%nucleotide 19\nnewpath\n'
                 r'\s+108.40 306.40 moveto\s+0 8 rlineto 8 0 rlineto 0 -8 rlineto closepath\n'
                 r'\s+0.50\d* 0.00\d* 0.00\d* 0.50\d* setcmykcolor', s):
    esl_itest.fail()
if not re.search(r'^%nucleotide 19\nnewpath\n'
                 r'\s+108.40 306.40 moveto  0 8 rlineto 8 0 rlineto 0 -8 rlineto closepath\n'
                 r'\s+0.92\d* 0.84\d* 0.00\d* 0.08\d* setcmykcolor',
                 s, flags=re.MULTILINE):
    esl_itest.fail()

# --no-ol       w/--indi, do not outline nts that are not most common nt
s = esl_itest.run(f'{easel} ssdraw --no-ol --indi {tmppfx}.sto {tmppfx}.t {tmppfx}.ps',
                  outfile=f'{tmppfx}.ps')
for testpattern in [ r'tRNA1', r'tRNA2', r'tRNA3', r'tRNA4', r'tRNA5',
                     r'\(A\) 180.00 392.00 moveto show',
                     r'\(Watson-Crick' ]:
    if not re.search(testpattern, s):
        esl_itest.fail(msg=f'with pattern {testpattern}')
if re.search(r'\(Positions !=', s):
    esl_itest.fail()
if not re.search(r'%nucleotide 19\nnewpath\n'
                 r'\s+108.40 306.40 moveto\s+0 8 rlineto 8 0 rlineto 0 -8 rlineto closepath\n'
                 r'\s+0.50\d* 0.00\d* 0.00\d* 0.50\d* setcmykcolor', s):
    esl_itest.fail()
if not re.search(r'^%nucleotide 19\nnewpath\n'
                 r'\s+108.40 306.40 moveto  0 8 rlineto 8 0 rlineto 0 -8 rlineto closepath\n'
                 r'\s+0.92\d* 0.84\d* 0.00\d* 0.08\d* setcmykcolor',
                 s, flags=re.MULTILINE):
    esl_itest.fail()

# --no-ntpp     w/--indi, do not draw nts on individual post prob diagrams
#               We're not actually checking here that residues are not drawn on PP diagrams
s = esl_itest.run(f'{easel} ssdraw --no-ntpp --indi {tmppfx}.sto {tmppfx}.t {tmppfx}.ps',
                  outfile=f'{tmppfx}.ps')
for testpattern in [ r'tRNA1', r'tRNA2', r'tRNA3', r'tRNA4', r'tRNA5',
                     r'\(A\) 180.00 392.00 moveto show',
                     r'\(Watson-Crick',
                     r'\(Positions !=' ]:
    if not re.search(testpattern, s):
        esl_itest.fail(msg=f'with pattern {testpattern}')
if not re.search(r'%nucleotide 19\nnewpath\n'
                 r'\s+108.40 306.40 moveto\s+0 8 rlineto 8 0 rlineto 0 -8 rlineto closepath\n'
                 r'\s+0.50\d* 0.00\d* 0.00\d* 0.50\d* setcmykcolor', s):
    esl_itest.fail()
if not re.search(r'^%nucleotide 19\nnewpath\n'
                 r'\s+108.40 306.40 moveto  0 8 rlineto 8 0 rlineto 0 -8 rlineto closepath\n'
                 r'\s+0.92\d* 0.84\d* 0.00\d* 0.08\d* setcmykcolor',
                 s, flags=re.MULTILINE):
    esl_itest.fail()

# Cleanup
for tmpfile in glob.glob(f'{tmppfx}.*'):
    os.remove(tmpfile)

print('ok')
